import axios from 'axios'

import * as TYPE from './types'
import * as statusesActions from './statuses'
import * as ticketsActions from './tickets'
import { error } from 'react-notification-system-redux'
import { ERRORS } from '../constants'
import errorTexts from '../errorTexts'

export const saveZoomInURL = (zoom) => {
  if (document.location.search) {
    const searchParams = new URLSearchParams(window.location.search)
    searchParams.set('zoom', zoom)
    const newParams = searchParams.toString()

    history.replaceState(null, null, `?${newParams}`)
  }
}

export const changeZoom = (delta) => {
  return (dispatch, getState) => {
    const state = getState()

    let zoom = +state.global.zoomOriginal + delta
    if (zoom < 0) zoom = 0
    if (zoom > 1) zoom = 1

    saveZoomInURL(zoom)

    dispatch({
      type: TYPE.CHANGE_ZOOM,
      payload: { zoom }
    })
  }
}

export const setZoom = (zoom) => {
  return (dispatch) => {
    saveZoomInURL(zoom)

    dispatch({
      type: TYPE.SET_ZOOM,
      payload: { zoom }
    })
  }
}

export const disableForm = () => ({
  type: TYPE.DISABLE_FORM
})

export const loadInitialData = (query, finalStatus) => async dispatch => {
  dispatch(statusesActions.fetchStatusesStarted())

  try {
    const response = await axios.get('/statuses')
    const { data, error } = response.data

    if (error) {
      dispatch(addErrorNotification(error))
      dispatch(statusesActions.fetchStatusesFail(error))
      dispatch(disableForm())
    }

    if (data) {
      dispatch(statusesActions.fetchStatusesSuccess(data))

      if (finalStatus) {
        dispatch(statusesActions.setDefaultFinalStatuses(finalStatus.map(item => data[item])))
      } else {
        const finalStatusData = Object.values(data).filter(sObj => sObj.key === 'closed' || sObj.key === 'resolved')
        finalStatus = finalStatusData.map(status => status.id)
        dispatch(statusesActions.setDefaultFinalStatuses(finalStatusData))
      }

      if (query !== undefined) {
        dispatch(ticketsActions.fetchTickets(query, finalStatus, true))
      }
    }
  } catch (e) {
    if (e.response && e.response.data && e.response.data.error) {
      dispatch(addErrorNotification(e.response.data.error))
    } else {
      console.error(e)
      dispatch(addErrorNotification(ERRORS.GENERIC_ERROR))
    }
    dispatch(statusesActions.fetchStatusesFail(e))
    dispatch(disableForm())
  }
}

export const addErrorNotification = (code) => (dispatch) => {
  dispatch(error({
    ...errorTexts(code),
    position: 'br',
    autoDismiss: 10
  }))
}
