import React from 'react'
import { connect } from 'react-redux'
import Notifications from 'react-notification-system-redux'

import Form from '../Form'
import DiagramWrapper from '../DiagramWrapper'
import Preloader from '../Preloader'

import {
  getTicketsStatus,
  getTicketsStatuses,
  getTicketList,
  getDateFrom,
  getDateTo,
  getIsWasRequest
} from '../../selectors/tickets'
import * as globalSelectors from '../../selectors/global'
import * as globalActions from '../../actions/global'
import * as queryActions from '../../actions/query'

import { dataFetchingStates } from '../../constants'

import s from './App.css'

const mapStateToProps = (state) => ({
  zoom: globalSelectors.getZoom(state),
  notifications: state.notifications,
  isInitialStatusesLoaded: state.statuses.isInitialStatusesLoaded,
  tickets: getTicketList(state),
  ticketsStatuses: getTicketsStatuses(state),
  ticketsStatus: getTicketsStatus(state),
  isWasRequest: getIsWasRequest(state),
  dateFrom: getDateFrom(state),
  dateTo: getDateTo(state)
})

const mapDispatchToProps = (dispatch) => {
  return {
    setTrackerQuery: (query) => {
      dispatch(queryActions.setTrackerQuery(query))
    },
    loadInitialData (query, defaultStatuses) {
      dispatch(globalActions.loadInitialData(query, defaultStatuses))
    }
  }
}

const getCookie = (name) => {
  const value = '; ' + document.cookie
  const parts = value.split('; ' + name + '=')
  if (parts.length === 2) return parts.pop().split(';').shift()
}

const appToken = getCookie('token')

if (!appToken) {
  const token = (/access_token=([^&]+)/.exec(document.location.hash) || [])[1]
  if (token) {
    document.cookie = `token=${token}; path=/`
    window.location.replace('/')
  }
}

class App extends React.Component {
  componentDidMount () {
    const { loadInitialData, setTrackerQuery } = this.props

    if (document.location.search) {
      const searchParams = new URLSearchParams(window.location.search)
      const query = searchParams.get('query')
      const finalStatuses = searchParams.get('finalStatuses')

      const normalizedStatuses = finalStatuses && finalStatuses.split(',').map(item => item)

      if (query) {
        setTrackerQuery(query)
      }

      loadInitialData(query, normalizedStatuses)
    } else {
      loadInitialData()
    }
  }

  render () {
    const {
      zoom,
      notifications,
      tickets,
      ticketsStatus,
      dateFrom,
      dateTo,
      isWasRequest,
      isInitialStatusesLoaded
    } = this.props

    return (
      <div className={s.root}>
        { appToken
          ? <a href="/logout" style={{ alignSelf: 'flex-end', marginRight: 'var(--space-l)', marginTop: 'var(--space-s)' }}>Выход</a>
          : <a href="/auth" style={{ alignSelf: 'flex-end', marginRight: 'var(--space-l)', marginTop: 'var(--space-s)' }}>Войти</a>
        }

        <Notifications notifications={notifications} />

        {isInitialStatusesLoaded &&
          <Form
            isrenderControls={tickets.length > 0 && ticketsStatus === dataFetchingStates.success && isWasRequest}
          />
        }

        {!isInitialStatusesLoaded && <Preloader className={s.root__preloader} isFullScreen />}

        {ticketsStatus === dataFetchingStates.loading && <Preloader className={s.root__preloader} />}

        {!tickets.length && isWasRequest && ticketsStatus === dataFetchingStates.success &&
          <div className={s.notFound}>По вашему запросу ничего не найдено</div>
        }

        {tickets.length > 0 && ticketsStatus === dataFetchingStates.success && isWasRequest &&
          <>
            <DiagramWrapper
              from={new Date(dateFrom)}
              to={new Date(dateTo)}
              zoom={zoom}
            />
          </>
        }

      </div>
    )
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(App)
