import React from 'react'
import { connect } from 'react-redux'
import throttle from 'raf-throttle'
import { scaleTime } from 'd3-scale'
import { DateTime } from 'luxon'

import Marker from '../Marker'
import Scale from '../Scale'
import LeftPanel from '../LeftPanel'
import TicketContainer from '../TicketContainer'
import MarkerPopup from '../MarkerPopup'
import TaskChangelogStatistic from '../TaskChangelogStatistic'
import TaskStatusesByTimeStatistic from '../TaskStatusesByTimeStatistic'
import Scrollbar from '../Scrollbar'
import join from '../../utils/join'
import { saveShiftToUrl } from '../../utils/saveShiftToUrl'
import { getInitialShift } from '../../utils/getInitialShift'
import { leftPanelWidth, appPadding } from '../../constants'

import { changeZoom } from '../../actions/global'

import s from './DiagramWrapper.css'

const mapDispatchToProps = {
  changeZoom
}

class DiagramWrapper extends React.Component {
  constructor (props) {
    super(props)
    this.widget = React.createRef()
    this.timeline = React.createRef()

    this.state = {
      originalScale: null,
      width: 0,
      mouse: {
        // x - calculated position for marker, not real mouse position
        x: null,
        y: null
      },
      rightBoundTimeline: -1,
      leftBoundTimeline: -1,
      bottomBoundWidget: -1,
      isMarkerNearRightBound: false,
      shift: getInitialShift(), // in ms
      ticketIdHovered: null,
      isInTicketsContainer: false,
      isInTimelineContainer: false,
      isDragging: false,
      dragPosition: null,
      isHover: false,
      scrollbar: {
        width: 0,
        offset: 0
      }
    }
  }

  componentDidMount () {
    this.init()
    window.addEventListener('resize', this.handleWindowResize)
    this.widget.current.addEventListener('wheel', this.onWheel, {
      passive: false
    })
  }

  componentDidUpdate (prevProps, prevState) {
    let originalScale

    if (prevState.width !== this.state.width) {
      this.updateBounds()
    }

    if (
      prevProps.width !== this.props.width ||
      prevProps.from !== this.props.from ||
      prevProps.to !== this.props.to
    ) {
      originalScale = this.calcOriginalScale()
      this.setState({
        originalScale
      })
    }

    if (
      prevProps.width !== this.props.width ||
      prevProps.zoom !== this.props.zoom ||
      prevProps.from !== this.props.from ||
      prevProps.to !== this.props.to ||
      prevState.shift !== this.state.shift
    ) {
      this.setState(this.getUpdatedScale({ originalScale }))
    }

    if (prevState.shift !== this.state.shift) {
      saveShiftToUrl(this.state.shift)
    }
  }

  componentWillUnmount () {
    window.removeEventListener('resize', this.handleWindowResize)
    this.widget.current.removeEventListener('wheel', this.onWheel)
    this.handleWindowResize.cancel()
    this.throttledMouseMove.cancel()
  }

  init = () => {
    const { width: widgetWidth, bottom } = this.widget.current.getBoundingClientRect()
    const width = widgetWidth - leftPanelWidth - 55
    const originalScale = this.calcOriginalScale({ width })
    this.setState({
      ...this.getUpdatedScale({ width, originalScale }),
      originalScale,
      width,
      mouse: {
        x: null
      },
      bottomBoundWidget: bottom
    }, () => {
      this.updateBounds()
    })
  };

  calcOriginalScale = (options = {}) => {
    const { from, to } = this.props
    const width = options.width || this.state.width

    return scaleTime()
      .domain([from, to])
      .range([0, width])
  }

  updateBounds = () => {
    if (!this.timeline.current) return
    const { left, right } = this.timeline.current.getBoundingClientRect()
    this.setState({
      rightBoundTimeline: right,
      leftBoundTimeline: left
    })
  }

  getUpdatedScale = (options = {}) => {
    const { from, to, zoom } = this.props
    const { shift } = this.state

    const width = options.width || this.state.width
    const originalScale = options.originalScale || this.state.originalScale

    const newState = {
      shiftedFrom: from,
      shiftedTo: to,
      scale: originalScale
    }

    if (zoom > 0) {
      const fromInMs = Number(from)
      const toInMs = Number(to)
      const durationInMs = Number(to) - Number(from)
      const pr = 0.4999 * zoom

      // смещение от from и от to после применения зума
      const zoomShift = durationInMs * pr
      const zoomedFromInMs = fromInMs + zoomShift
      const zoomedToInMs = toInMs - zoomShift
      const zoomedDurationInMs = zoomedToInMs - zoomedFromInMs

      let shiftedFromInMs = zoomedFromInMs + shift
      let shiftedToInMs = zoomedToInMs + shift

      if (shiftedFromInMs < fromInMs) {
        shiftedFromInMs = fromInMs
        shiftedToInMs = fromInMs + zoomedDurationInMs
        newState.shift = fromInMs - zoomedFromInMs
      }

      if (shiftedToInMs > toInMs) {
        shiftedFromInMs = toInMs - zoomedDurationInMs
        shiftedToInMs = toInMs
        newState.shift = (toInMs - zoomedDurationInMs) - zoomedFromInMs
      }

      newState.shiftedFrom = new Date(shiftedFromInMs)
      newState.shiftedTo = new Date(shiftedToInMs)

      newState.scale = scaleTime()
        .domain([newState.shiftedFrom, newState.shiftedTo])
        .range([0, width])
    } else {
      newState.shift = 0
    }

    newState.scrollbar = {
      width: originalScale(newState.shiftedTo) - originalScale(newState.shiftedFrom),
      offset: originalScale(newState.shiftedFrom)
    }

    return newState
  }

  handleWindowResize = throttle(() => {
    this.init()
  });

  onMouseDown = e => {
    this.setState({
      isDragging: true,
      dragPosition: e.clientX
    })
  }

  onMouseUp = () => {
    this.setState({
      isDragging: false,
      dragPosition: null
    })
  }

  onMouseMove = e => {
    const { clientX, clientY, target } = e
    this.throttledMouseMove({ clientX, clientY, target })
  }

  throttledMouseMove = throttle(({ clientX, clientY, target }) => {
    const {
      isHover,
      rightBoundTimeline,
      leftBoundTimeline,
      isDragging,
      dragPosition,
      scale
    } = this.state

    if (!isHover) return

    let ticketIdHovered
    let change = 0

    const closestTicketLine = target.closest('.ticket-line')
    if (closestTicketLine) {
      ticketIdHovered = closestTicketLine.getAttribute('data-ticket-id')
    }

    if (clientX < leftBoundTimeline) {
      clientX = leftBoundTimeline
    }
    if (clientX > rightBoundTimeline) {
      clientX = rightBoundTimeline
    }

    if (isDragging) {
      change = scale.invert(dragPosition) - scale.invert(clientX)
    }

    this.setState(s => ({
      shift: s.shift + change,
      dragPosition: clientX,
      mouse: {
        x: clientX - appPadding,
        y: clientY
      },
      ticketIdHovered: ticketIdHovered || s.ticketIdHovered,
      isInTicketsContainer: Boolean(target.closest('#tickets-container')),
      isInTimelineContainer: Boolean(target.closest('#timeline'))
    }))
  })

  onWheel = e => {
    e.preventDefault()
    e.stopPropagation()
    this.pan({
      ctrlKey: e.ctrlKey,
      deltaX: e.deltaX,
      deltaY: e.deltaY
    })
  }

  pan = throttle(({ ctrlKey, deltaX, deltaY }) => {
    const { scale } = this.state

    if (ctrlKey) {
      this.props.changeZoom(-deltaY * 0.001)
      return
    }

    this.widget.current.scrollTop += deltaY

    this.setState(state => ({
      shift: state.shift + (scale.invert(deltaX) - state.shiftedFrom),
      ...this.getUpdatedScale()
    }))
  })

  onScroll = throttle(() => {
    this.resetPopupVisibility()
  })

  onMouseEnter = () => {
    this.setState({
      isHover: true
    })
  }

  onMouseLeave = () => {
    this.resetPopupVisibility()
    this.setState({
      isHover: false
    })
  }

  handleScrollbarChange = shift => {
    this.setState({ shift })
  }

  resetPopupVisibility = () => {
    const { isInTicketsContainer, isInTimelineContainer } = this.state

    if (!isInTicketsContainer && !isInTimelineContainer) return
    this.setState({
      isInTicketsContainer: false,
      isInTimelineContainer: false
    })
  }

  render () {
    const {
      width,
      shiftedFrom: from,
      shiftedTo: to,
      mouse,
      originalScale,
      scale,
      ticketIdHovered,
      isInTicketsContainer,
      isInTimelineContainer,
      isDragging,
      scrollbar
    } = this.state

    if (!width || !scale) {
      return (
        <div ref={this.widget} className={s.widget} />
      )
    }

    const scaleWidth = width
    const hasWidgetVerticalScroll = this.widget.current.offsetWidth > this.widget.current.clientWidth
    const markerTime = scale.invert(mouse.x - leftPanelWidth)

    return (
      <>
        <div
          ref={this.widget}
          className={join(s.widget)}
          onMouseDown={this.onMouseDown}
          onMouseMove={this.onMouseMove}
          onMouseUp={this.onMouseUp}
          onScroll={this.onScroll}
          onMouseEnter={this.onMouseEnter}
          onMouseLeave={this.onMouseLeave}
        >
          <Scale
            scale={scale}
            scaleWidth={scaleWidth}
            from={from}
            to={to}
            scaleRef={this.timeline}
          />
          <Marker
            hasWidgetVerticalScroll={hasWidgetVerticalScroll}
            position={mouse.x}
            scale={scale}
          />
          <MarkerPopup
            position={{
              x: mouse.x + appPadding,
              y: mouse.y
            }}
            boundaries={{
              right: this.state.rightBoundTimeline,
              bottom: this.state.bottomBoundWidget
            }}
            markerTime={DateTime.fromJSDate(markerTime).setLocale('ru-Ru').toFormat('HH:mm d MMM yyyy')}
            ticketId={ticketIdHovered}
          >
            {isInTicketsContainer &&
              <TaskChangelogStatistic
                to={this.props.to}
                ticketIdHovered={ticketIdHovered}
              />
            }
            {isInTimelineContainer &&
              <TaskStatusesByTimeStatistic markerTime={DateTime.fromJSDate(markerTime).toMillis()} />
            }
          </MarkerPopup>
          <LeftPanel />
          <TicketContainer
            scale={scale}
            to={this.props.to}
            scaleWidth={scaleWidth}
            isDragging={isDragging}
          />
          {/* <code style={{ position: 'fixed' }}><pre>{JSON.stringify(this.state, null, 2)}</pre></code> */}
        </div>
        <div className={s.footer}>
          <Scrollbar
            from={this.props.from}
            shiftedFrom={from}
            originalScale={originalScale}
            shift={this.state.shift}
            width={scrollbar.width}
            offset={scrollbar.offset}
            onChange={this.handleScrollbarChange}
          />
        </div>
      </>
    )
  }
}

export default connect(null, mapDispatchToProps)(DiagramWrapper)
