import React from 'react'
import { connect } from 'react-redux'
import * as statusSelectors from '../../selectors/statuses'
import * as querySelectors from '../../selectors/query'
import * as ticketsSelectors from '../../selectors/tickets'
import join from '../../utils/join'
import Button from '../Button'
import QueryInput from '../QueryInput'
import MultiSelect from '../Select'
import Controls from '../Controls'

import s from './Form.css'
import { setTrackerQuery } from '../../actions/query'
import { fetchTickets } from '../../actions/tickets'
import { setFinalStatuses } from '../../actions/statuses'

import { dataFetchingStates } from '../../constants'

const mapStateToProps = (state) => ({
  trackerQuery: querySelectors.getTrackerQuery(state),
  queryError: querySelectors.getQueryError(state),
  statusesForSelect: statusSelectors.getStatusesForSelect(state),
  finalStatusesForSelect: statusSelectors.getFinalStatusesForSelect(state),
  statuses: state.statuses.statuses,
  ticketsStatus: ticketsSelectors.getTicketsStatus(state)
})

const mapDispatchToProps = (dispatch) => ({
  fetchTickets: (query, status) => {
    dispatch(fetchTickets(query, status))
  },
  setTrackerQuery: (query) => {
    dispatch(setTrackerQuery(query))
  },
  setFinalStatuses: finalStatusesArr => {
    dispatch(setFinalStatuses(finalStatusesArr))
  }
})

class Form extends React.Component {
  state = {
    finalStatusesForSelect: []
  }

  componentDidMount () {
    this.setState({
      finalStatusesForSelect: this.props.finalStatusesForSelect
    })
  }

  handleKeyPress = e => {
    if (e.key === 'Enter') {
      this.handleSubmit()
    }
  }

  handleChange = e => {
    this.props.setTrackerQuery(e.currentTarget.value)
  }

  handleFinalStatusesChange = selectedArr => {
    this.setState({ finalStatusesForSelect: selectedArr || [] })
  }

  resetQuery = () => {
    this.props.setTrackerQuery('')
  }

  handleSubmit = () => {
    const { trackerQuery, fetchTickets } = this.props
    const { finalStatusesForSelect } = this.state

    const finalStatusesIds = finalStatusesForSelect.length > 0 ? finalStatusesForSelect.map(item => item.id) : []

    this.props.setFinalStatuses(finalStatusesForSelect)

    const query = `query=${encodeURIComponent(trackerQuery || '')}`
    const status = `finalStatuses=${encodeURIComponent((finalStatusesIds || []).join(','))}`

    const queryParams = `?${query}&${status}`

    history.pushState(null, null, queryParams)

    fetchTickets(trackerQuery, finalStatusesIds)
  }

  render () {
    const {
      trackerQuery,
      queryError,
      className = '',
      statusesForSelect,
      isrenderControls,
      ticketsStatus
    } = this.props

    const { finalStatusesForSelect } = this.state

    const isFormDisabled = ticketsStatus === dataFetchingStates.loading ||
      ticketsStatus === dataFetchingStates.disabled

    return (
      <div className={join(s.TicketSearch, className)}>
        <div className={s.Row}>
          <div className={s.RowQuery}>
            <QueryInput
              value={trackerQuery}
              error={queryError}
              disabled={isFormDisabled}
              onChange={this.handleChange}
              onKeyPress={this.handleKeyPress}
              onClearPressed={this.resetQuery}
            />
          </div>

          <div className={s.RowSelect}>
            <MultiSelect
              options={statusesForSelect}
              value={finalStatusesForSelect}
              disabled={isFormDisabled}
              onChange={this.handleFinalStatusesChange}
            />
          </div>
        </div>

        <div className={s.flexLine}>
          <Button
            type="Accent"
            disabled={isFormDisabled}
            className={s.TicketSearch__button}
            onClick={this.handleSubmit}
          >
            Применить
          </Button>

          {isrenderControls && <Controls />}
        </div>
      </div>
    )
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(Form)
