import { DateTime } from 'luxon'
import { INTERVALS, MIN_SIZE } from '../constants'

export function getFormatter (scale, formatCallback) {
  let savedPrev = {}
  return (date, isBoundDate = false) => {
    date = DateTime.fromJSDate(date)
    const tickObject = formatCallback(date, isBoundDate)
    const prev = savedPrev
    savedPrev = { ...tickObject }
    tickObject.date = date
    tickObject.position = Math.round(scale(date))
    tickObject.key = [tickObject.top, tickObject.middle, tickObject.bottom].join()
    // удаляем повторяющиеся значения, если подпись для текущего тика равна предыдущей
    if (tickObject.top === prev.top) {
      delete tickObject.top
    }
    if (tickObject.middle === prev.middle) {
      delete tickObject.middle
    }
    return tickObject
  }
}

const TO_FORMAT = new Map([
  [INTERVALS.m5, (scale) => getFormatter(scale, date => ({
    interval: 'm5',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.m15, (scale) => getFormatter(scale, date => ({
    interval: 'm15',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.m30, (scale) => getFormatter(scale, date => ({
    interval: 'm30',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.h, (scale) => getFormatter(scale, date => ({
    interval: 'h',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.h6, (scale) => getFormatter(scale, date => ({
    interval: 'h6',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.h12, (scale) => getFormatter(scale, date => ({
    interval: 'h12',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: date.toFormat('HH:mm')
  }))],
  [INTERVALS.d, (scale) => getFormatter(scale, (date, isBoundDate) => ({
    interval: 'd',
    top: date.toFormat('LLLL'),
    middle: date.toFormat('d'),
    bottom: isBoundDate ? date.toFormat('HH:mm') : date.toFormat('EEE')
  }))],
  [INTERVALS.w, (scale) => getFormatter(scale, (date, isBoundDate) => ({
    interval: 'w',
    top: date.toFormat('yyyy'),
    middle: date.toFormat('LLLL'),
    bottom: isBoundDate ? date.toFormat('d') : date.toFormat('d') + '–' + date.plus({ days: 7 }).toFormat('d')
  }))],
  [INTERVALS.m, (scale) => getFormatter(scale, date => ({
    interval: 'm',
    top: date.toFormat('yyyy'),
    middle: date.toFormat('LLLL'),
    bottom: ''
  }))],
  [INTERVALS.y, (scale) => getFormatter(scale, date => ({
    interval: 'y',
    top: '',
    middle: date.toFormat('yyyy'),
    bottom: ''
  }))]
])

export function getTicks (scale, from, to) {
  let ticks
  let format
  for (const interval of Object.values(INTERVALS)) {
    const nextTicks = scale.ticks(interval)
    if (nextTicks.length < 1) {
      break
    }
    ticks = nextTicks
    format = TO_FORMAT.get(interval)(scale)
    const [first, second] = ticks
    const firstInPx = scale(first)
    const secondInPx = scale(second)

    // растояние достаточное, чтобы разместить подпись, выходим из цикла
    // с подобранным интервалом
    if (secondInPx - firstInPx >= MIN_SIZE.get(interval)) {
      break
    }
  }
  // не смогли подобрать подходящий интервал
  if (!format && !ticks) {
    format = TO_FORMAT.get(INTERVALS.m5)(scale)
    return [
      format(from, true),
      format(to),
      format(to, true)
    ]
  }
  return [
    format(from, true),
    ...ticks.map(tickDate => format(tickDate)),
    format(to, true)
  ]
}
