const cookie = require('cookie')
const sentry = require('@sentry/node')
const QuickLRU = require('quick-lru')

const config = require('./config')
const { version } = require('./package.json')

const Server = require('./modules/httpserver')
const CacheWarmup = require('./modules/cacheWarmup')
const Queue = require('./modules/queue')
const TrackerApi = require('./modules/trackerApi')
const TaskList = require('./modules/taskList')
const Statuses = require('./modules/statuses')

const SYNC_CONFIG = config.get('backgroundSync')
const PORT = config.get('port')
const HOST = config.get('host')
const { dsn: SENTRY_DSN } = config.get('sentry')
const CHANGELOG_PARALLEL_DOWNLOADS_COUNT = 15
const RETRY_ATTEMPTS_ON_ERROR = 1
const CHANGELOGS_CACHE_SIZE = 2000

sentry.init({ dsn: SENTRY_DSN, release: version })

const server = Server({ sentry })
const trackerApi = new TrackerApi({
  logger: server.log,
  sentry,
  orgId: config.get('orgId'),
  baseUri: config.get('trackerApiUri')
})
const queue = new Queue({
  concurency: CHANGELOG_PARALLEL_DOWNLOADS_COUNT,
  attempts: RETRY_ATTEMPTS_ON_ERROR,
  rejectOnlyOnFailsInARow: true
})
const cache = new QuickLRU({ maxSize: CHANGELOGS_CACHE_SIZE })
const taskList = new TaskList({ trackerApi, queue, cache })
const statuses = new Statuses({ trackerApi })
const cacheWarmup = new CacheWarmup({ ...SYNC_CONFIG, taskList })

const authHandler = require('./handlers/auth')()
const logoutHandler = require('./handlers/logout')({ cacheWarmup })
const infoHandler = require('./handlers/info')({ sentry, version })

const tasksListHandler = require('./handlers/taskList')({ taskList })
const statusHandler = require('./handlers/statuses')({ statuses })

server
  .addHook('onRequest', (req, res, next) => {
    const { token } = cookie.parse(req.raw.headers.cookie || '')
    if (token) {
      req.token = token
      cacheWarmup.followToken(token)
    }
    next()
  })

  .staticServe('../client/dist', '/')

  .get('/auth', authHandler)
  .get('/logout', logoutHandler)

  .post('/task-list', tasksListHandler)
  .get('/statuses', statusHandler)
  .get('/info', infoHandler)

  .listen(PORT, HOST)
