const fastify = require('fastify')
const corsPlugin = require('fastify-cors')
const staticPlugin = require('fastify-static')

const fs = require('fs')
const path = require('path')

const ERROR_CODES = require('./errorCodes')

const Server = ({
  bodyLimit = 1024,
  sentry
} = {}) => {
  const app = fastify({
    bodyLimit,
    modifyCoreObjects: false,
    disableRequestLogging: true,
    logger: {
      prettyPrint: {
        translateTime: 'SYS:dd/mm/yyyy HH:MM:ss',
        ignore: 'pid,hostname'
      }
    }
  })

  app.register(corsPlugin)

  app.setNotFoundHandler((req, res) => {
    res.code(404).send({ error: ERROR_CODES.NOT_FOUND, data: null })
  })

  app.setErrorHandler(async (error, req, res) => {
    sentry.captureException(error)
    app.log.error(error)

    const code = ERROR_CODES[error.code] ||
      (error.statusCode === 404 && ERROR_CODES.NOT_FOUND) ||
      error.code ||
      ERROR_CODES.UNEXPECTED_ERROR
    res.code(error.statusCode || 500)

    return { error: code, data: null }
  })

  app.staticServe = (root, url) => {
    const baseRoot = path.join(path.dirname(module.parent.filename), root)

    app.register(staticPlugin, {
      serve: false,
      root: baseRoot
    })

    app.file = (pattern, filePath) => app.get(pattern, (req, res) => {
      res.sendFile(filePath)
    })

    app.folder = (prefix, root = '') => {
      prefix = prefix.replace(/\/+$/, '')
      root = root.replace(/(^\/+|\/+$)/, '')

      fs.readdirSync(baseRoot + '/' + root).forEach(file => {
        const filePath = root + '/' + file
        const fileUrl = prefix + '/' + file

        if (fs.statSync(baseRoot + '/' + filePath).isDirectory()) {
          app.folder(fileUrl, filePath)
        } else {
          if (file === 'index.html' || file === 'index.htm') {
            app.file(prefix + '/', filePath)
          }
          app.file(fileUrl, filePath)
        }
      })
      return app
    }

    if (url) { app.folder(url) }

    return app
  }

  const _listen = app.listen.bind(app)

  app.listen = (...args) => {
    const promise = _listen(...args)
    if (!promise || !promise.then) { return }

    promise.catch(err => {
      app.log.error(err)
    })
  }

  return app
}

module.exports = Server
