const PrioritizedHashQueue = require('./prioritizedHashQueue')
const deferredPromise = require('./deferredPromise')

class Queue {
  constructor ({
    concurency = Infinity,
    attempts = 0,
    rejectOnlyOnFailsInARow = false
  } = {}) {
    this._queue = new PrioritizedHashQueue()
    this._deferredPromises = {}

    this._concurrencyRuns = 0
    this._concurrencyLimit = concurency
    this._attempts = attempts
    this._rejectOnlyOnFailsInARow = rejectOnlyOnFailsInARow

    this._inARowFailsCount = 0
  }

  run (id, priority, task) {
    if (!this._deferredPromises[id]) {
      this._deferredPromises[id] = deferredPromise()
      this._deferredPromises[id].active = false
      this._deferredPromises[id].attempts = this._attempts
    }

    if (this._queue.place(id, priority, task)) {
      this._next()
    }

    return this._deferredPromises[id].promise
  }

  _next () {
    if (this._concurrencyRuns === this._concurrencyLimit) { return }
    const [id, priority, task] = this._queue.pop()
    if (!id) { return }
    if (!this._deferredPromises[id] || this._deferredPromises[id].active) { return this._next() }

    this._deferredPromises[id].active = true
    this._concurrencyRuns++

    Promise.resolve().then(task)
      .then(result => {
        this._inARowFailsCount = 0
        this._deferredPromises[id].resolve(result)
        delete this._deferredPromises[id]
      })
      .catch((err) => {
        if (this._deferredPromises[id].attempts > 0) {
          this._deferredPromises[id].attempts--
          this._deferredPromises[id].active = false
          this._queue.place(id, priority, task)
        } else if (this._rejectOnlyOnFailsInARow && (++this._inARowFailsCount < this._concurrencyRuns)) {
          this._deferredPromises[id].active = false
          this._queue.place(id, priority, task)
        } else {
          this._deferredPromises[id].reject(err)
          delete this._deferredPromises[id]
        }
      })
      .finally(() => {
        this._concurrencyRuns--
        this._next()
      })
  }
}

module.exports = Queue
