const changelogFormatter = require('./changelogFormatter')
const foldChangelogFinalStatuses = require('./foldChangelogFinalStatus')
const { pickBaseTaskFields, formatTaskForClient } = require('./taskFormatter')
const isUpToDate = require('./isCachedUpToDate')
const sortByChangelog = require('./sortByChangelog')
const filterWithoutEmpty = require('../../utils/filterWithoutEmpty')

const { formatTrackerApiError } = require('../../dependences/errorCodes')

const TASKS_COUNT_PER_SCROLL = 500

const getTaskChangelog = (trackerApi, cache, token, id) => trackerApi
  .getTaskChangelog(token, id)
  .then(changelogFormatter)
  .then(changelog => changelog.sort((a, b) => a.updatedAt - b.updatedAt))
  .then(changelog => {
    cache.set(id, changelog)
    return changelog
  })

async function getAllTasks (trackerApi, queue, token, query) {
  const tasks = []
  try {
    let uri = null
    do {
      const next = await queue.run(JSON.stringify({ token, query }), 1,
        () => trackerApi.getTaskList(token, query, uri, TASKS_COUNT_PER_SCROLL)
      )
      const group = next.group
      uri = next.nextPageUri
      tasks.push(...group.map(pickBaseTaskFields))
    }
    while (uri)
  } catch (err) {
    return [err, tasks]
  }

  return [null, tasks]
}

class TaskList {
  constructor ({ trackerApi, cache, queue }) {
    this.queue = queue
    this.cache = cache
    this.trackerApi = trackerApi
  }

  async load (token, query, finalStatus, priority = 2) {
    const [getTasksError, tasks] = await getAllTasks(this.trackerApi, this.queue, token, query)

    const cachedTasks = []
    const tasksForChanglogDownload = tasks.filter(task => {
      const cached = this.cache.get(task.id)
      if (!cached) {
        return true
      } else if (isUpToDate(task, cached)) {
        const changelog = foldChangelogFinalStatuses(cached, finalStatus)
        cachedTasks.push(formatTaskForClient(task, changelog))
        return false
      } else {
        this.cache.delete(task.id)
        return true
      }
    })

    let getTaskChangelogError
    const promises = tasksForChanglogDownload.map(task => this.queue
      .run(task.id, priority, () => getTaskChangelog(this.trackerApi, this.cache, token, task.id, finalStatus))
      .then(changelog => foldChangelogFinalStatuses(changelog, finalStatus))
      .then(changelog => formatTaskForClient(task, changelog))
      .catch(err => {
        if (!getTaskChangelogError) { getTaskChangelogError = err }
        return null
      })
    )

    const trackerError = getTasksError || getTaskChangelogError
    const error = trackerError ? formatTrackerApiError(trackerError) : null
    const tasksWithChangelog = filterWithoutEmpty(await Promise.all(promises))

    const sortedTasks = sortByChangelog(cachedTasks.concat(tasksWithChangelog), finalStatus)

    return [error, sortedTasks]
  }
}

module.exports = TaskList
