const request = require('request-promise-native')
const cursorLinksParser = require('./cursorLinksParser')

class TrackerApi {
  constructor ({ logger, sentry, orgId, baseUri }) {
    this._baseUri = baseUri
    this._request = async (token, uri, body = null) => request({
      method: body ? 'POST' : 'GET',
      body,
      uri,
      headers: {
        Authorization: `OAuth ${token}`,
        'X-Org-Id': `${orgId}`
      },
      json: true,
      resolveWithFullResponse: true
    })
      .catch(err => {
        if (err.statusCode >= 500) {
          logger.error(err)
          sentry.captureException(err)
        }
        throw err
      })
  }

  async getStatuses (token) {
    const response = await this._request(token, `${this._baseUri}statuses`)
    return response.body
  }

  async getTaskList (token, query = '', scrollUri = null, perScroll = 1000) {
    const uri = scrollUri || `${this._baseUri}issues/_search?scrollType=unsorted&perScroll=${perScroll}`
    const response = await this._request(token, uri, { query })
    const nextPageUri = cursorLinksParser(response.headers.link).next
    return {
      group: response.body,
      nextPageUri
    }
  }

  async getTaskChangelog (token, id) {
    const response = await this._request(token, `${this._baseUri}issues/${id}/changelog?type=IssueWorkflow`)
    return response.body
  }
}

module.exports = TrackerApi
