package ru.yandex.direct.intapi;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.controller.BalanceClientController;
import ru.yandex.direct.intapi.entity.balanceclient.exception.BalanceClientException;

/**
 * Обработчик исключений для BalanceClientController
 * <p>
 * Помечен {@link Ordered#HIGHEST_PRECEDENCE},
 * чтобы обработчик был выше чем {@link IntApiExceptionHandlers}
 *
 * @see BalanceClientController
 * @see IntApiExceptionHandlers
 */
@Order(Ordered.HIGHEST_PRECEDENCE)
@ControllerAdvice(assignableTypes = BalanceClientController.class)
public class BalanceClientExceptionHandlers {

    private static final Logger logger = LoggerFactory.getLogger(BalanceClientExceptionHandlers.class);
    private static final BalanceClientResponse INTERNAL_SERVER_ERROR =
            BalanceClientResponse.criticalError("Internal server error");

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    @ResponseBody
    public BalanceClientResponse argumentTypeMismatchHandling(MethodArgumentTypeMismatchException exception) {
        logException(exception);
        return BalanceClientResponse.criticalError("argument type mismatch");
    }

    @ExceptionHandler(BalanceClientException.class)
    @ResponseBody
    public BalanceClientResponse balanceClientException(BalanceClientException exception) {
        logException(exception);
        return exception.getResponse();
    }

    @ExceptionHandler(RuntimeException.class)
    @ResponseBody
    public BalanceClientResponse internalException(RuntimeException exception) {
        logException(exception);
        return INTERNAL_SERVER_ERROR;
    }

    private void logException(Throwable t) {
        logger.error("Exception during intapi request", t);
    }
}
