package ru.yandex.direct.intapi;

import com.fasterxml.jackson.databind.JsonMappingException;
import org.eclipse.jetty.io.EofException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.CommonTranslations;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.exception.WebBadRequestException;

import static ru.yandex.direct.utils.JsonUtils.toJson;

@ControllerAdvice
public class IntApiExceptionHandlers {
    private static final Logger logger = LoggerFactory.getLogger(
            IntApiExceptionHandlers.class);
    private final TranslationService translationService;

    @Autowired
    public IntApiExceptionHandlers(TranslationService translationService) {
        this.translationService = translationService;
    }

    @ExceptionHandler(IntApiException.class)
    public ResponseEntity<String> intApiExceptionHandler(IntApiException exception) {
        logException(exception);
        return responseWithJson(exception.getErrorObject(), exception.getHttpStatus());
    }

    @ExceptionHandler(WebBadRequestException.class)
    public ResponseEntity<String> webBadRequestExceptionHandler(WebBadRequestException exception) {
        logException(exception);
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, exception.getMessage());
        return responseWithJson(response, HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler(AccessDeniedException.class)
    public ResponseEntity<String> accessDeniedExceptionHandler(AccessDeniedException exception) {
        logException(exception);
        return responseWithJson(exception.getCode() + " : " + exception.getMessage(), HttpStatus.FORBIDDEN);
    }

    @ExceptionHandler(TranslatableException.class)
    public ResponseEntity<String> defaultTranslatableExceptionHandler(TranslatableException exception) {
        logException(exception);
        return responseWithJson(exception.getCode() + " : " + exception.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ResponseEntity<String> jsonMappingErrorHandler(HttpMessageNotReadableException exception) {
        logException(exception);
        Throwable cause = exception.getCause();
        String msg = "";
        if (cause != null && cause instanceof JsonMappingException) {
            JsonMappingException jsonMappingException = (JsonMappingException) cause;
            msg = JsonUtils.extractJsonMappingErrorLocation(jsonMappingException);
        }
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, msg);
        return responseWithJson(response, HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({
            // Missing parameter -- это если не указали какой-либо из праметров, ожидаемых в URL#queryString
            MissingServletRequestParameterException.class,
            // Argument type mismatch -- это если параметр задан, но не может быть сконвертирован в нужный тип
            MethodArgumentTypeMismatchException.class
    })
    public ResponseEntity<String> missingRequestParameterHandler(Exception exception) {
        logException(exception);
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, exception.getMessage());
        return responseWithJson(response, HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler(
            //HttpMediaTypeNotSupportedException - если, к примеру, ждем 'application/json',
            //но пришел 'application/x-www-form-urlencoded'(такой content-type делает curl по-умолчанию)
            HttpMediaTypeNotSupportedException.class
    )
    public ResponseEntity<String> wrongHttpMediaTypeHandler(Exception exception) {
        logException(exception);
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, exception.getMessage());
        return responseWithJson(response, HttpStatus.BAD_REQUEST);
    }

    //В spring также есть DataAccessException, так что здесь явное указание класса - чтобы не перепутать
    @ExceptionHandler(org.jooq.exception.DataAccessException.class)
    public ResponseEntity<String> dataAccessExceptionHandler(Exception exception) {
        logException(exception);
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.INTERNAL_ERROR, "db-related problem");
        return responseWithJson(response, HttpStatus.INTERNAL_SERVER_ERROR);
    }

    @ExceptionHandler(EofException.class)
    @ResponseBody
    public ResponseEntity<String> eofErrorHandler(Exception e) {
        // Jetty EOF -- это QuietException, который обозначает разрыв соединения с клиентской стороны. Логируем как INFO
        logger.info("eofException", e);
        return ResponseEntity.status(ru.yandex.misc.io.http.HttpStatus.SC_499_CLIENT_CLOSED_REQUEST).build();
    }

    @ExceptionHandler(Exception.class)
    public ResponseEntity<String> defaultExceptionHandler(Exception exception) {
        logException(exception);
        ErrorResponse response = new ErrorResponse(ErrorResponse.ErrorCode.INTERNAL_ERROR,
                translationService.translate(CommonTranslations.INSTANCE.serviceUnavailable()));
        return responseWithJson(response, HttpStatus.INTERNAL_SERVER_ERROR);
    }

    private void logException(Throwable t) {
        logger.error("Exception during intapi request", t);
    }

    private ResponseEntity<String> responseWithJson(Object bodyObject, HttpStatus status) {
        return new ResponseEntity<>(toJson(bodyObject), jsonHeaders(), status);
    }

    private HttpHeaders jsonHeaders() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        return headers;
    }
}
