package ru.yandex.direct.intapi.common.converter.directtsv;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.MediaType;
import org.springframework.http.converter.AbstractHttpMessageConverter;

import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер, сериализующий коллекцию объектов в формат TSV перлового интапи директа.
 * Сначала превращает объект в Map с помощью jackson, а затем сохраняет этот Map как ряд данных.
 * Не умеет работать с Map с более чем одним уровнем вложенности
 */
public class DirectTsvMessageConverter extends AbstractHttpMessageConverter<Object> {
    public static final String DIRECT_TSV_MEDIA_TYPE = "application/direct-tsv";
    private static final String DELIMITER = "\t";

    public DirectTsvMessageConverter() {
        super(MediaType.valueOf(DIRECT_TSV_MEDIA_TYPE));
    }

    @Override
    protected boolean supports(Class<?> clazz) {
        return true; //Возвращаем true т.к. действительно хотим здесь обрабатывать все типы
    }

    @Override
    protected Object readInternal(Class<?> clazz, HttpInputMessage inputMessage)
            throws IOException {
        throw new UnsupportedOperationException("Content type cannot be read");
    }

    @Override
    protected void writeInternal(Object obj, HttpOutputMessage outputMessage)
            throws IOException {
        outputMessage.getHeaders()
                .setContentType(new MediaType(outputMessage.getHeaders().getContentType(), StandardCharsets.UTF_8));
        if (obj instanceof Collection) {
            writeTsv((Collection<?>) obj, outputMessage);
        } else {
            // Ошибка может прийти в строке, другие типы данных просто пробрасываем
            outputMessage.getBody().write(obj.toString().getBytes());
        }
    }

    void writeTsv(Collection<?> data, HttpOutputMessage outputMessage)
            throws IOException {
        List<String> keysOrder = null;
        BiMap<String, String> keyToName = HashBiMap.create();
        OutputStream bodyStream = outputMessage.getBody();

        if (!data.isEmpty()) {
            Object o = data.toArray()[0];

            for (Method method : o.getClass().getMethods()) {
                DirectTsvColumn tc = method.getAnnotation(DirectTsvColumn.class);
                if (tc != null) {
                    JsonProperty jp = method.getAnnotation(JsonProperty.class);
                    if (jp != null) {
                        keyToName.put(tc.value(), jp.value());
                    }
                }
            }

            DirectTsvFieldsOrder tfo = o.getClass().getAnnotation(DirectTsvFieldsOrder.class);
            if (tfo != null) {
                bodyStream.write(getKeysTsvString(tfo.value()).getBytes(StandardCharsets.UTF_8));
                keysOrder = getKeysOrder(Arrays.asList(tfo.value()), keyToName);
            }
        }

        List<Map<String, Object>> responseRepr = getResponseRepresentation(data);
        if (!responseRepr.isEmpty() && keysOrder == null) {
            Map<String, Object> map = responseRepr.get(0);
            keysOrder = new ArrayList<>(map.keySet());
            List<String> headerKeyOrder = getKeysOrder(keysOrder, keyToName.inverse());
            bodyStream.write(getKeysTsvString(headerKeyOrder).getBytes(StandardCharsets.UTF_8));
        }

        for (Map<String, Object> map : responseRepr) {
            bodyStream.write(getTsvString(map, keysOrder).getBytes(StandardCharsets.UTF_8));
        }
        bodyStream.write("#End\n".getBytes(StandardCharsets.UTF_8));
    }

    List<String> getKeysOrder(Collection<String> keys, Map<String, String> keyToName) {
        List<String> keysOrder = new ArrayList<>();
        for (String key : keys) {
            keysOrder.add(keyToName.getOrDefault(key, key));
        }
        return keysOrder;
    }

    private List<Map<String, Object>> getResponseRepresentation(Collection data) {
        TypeFactory tf = JsonUtils.getTypeFactory();
        JavaType javaType =
                tf.constructCollectionType(List.class, tf.constructMapType(Map.class, String.class, Object.class));

        return JsonUtils.getObjectMapper().convertValue(data, javaType);
    }

    String getTsvString(Map<String, Object> map, List<String> keysOrder) {
        List<Object> stringParts = new ArrayList<>();
        for (String key : keysOrder) {
            stringParts.add(nvl(map.get(key), ""));
        }
        return String.join(DELIMITER, mapList(stringParts, Object::toString)) + "\n";
    }

    String getKeysTsvString(List<String> keysOrder) {
        return "#" + String.join(DELIMITER, keysOrder) + "\n";
    }

    String getKeysTsvString(String... keysOrder) {
        return getKeysTsvString(Arrays.asList(keysOrder));
    }
}

