package ru.yandex.direct.intapi.configuration;

import java.util.Arrays;
import java.util.List;

import javax.servlet.Filter;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.protobuf.ProtobufHttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.web.filter.CompositeFilter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import ru.yandex.direct.common.logging.LoggingConfigurerInterceptor;
import ru.yandex.direct.common.logging.LoggingSettings;
import ru.yandex.direct.common.metrics.MetricsFilter;
import ru.yandex.direct.common.metrics.MetricsInterceptor;
import ru.yandex.direct.common.tracing.TraceContextFilter;
import ru.yandex.direct.intapi.common.converter.directtsv.DirectTsvMessageConverter;
import ru.yandex.direct.intapi.logging.IntapiLoggingFilter;
import ru.yandex.direct.intapi.webapp.semaphore.IntapiSemaphoreInterceptor;
import ru.yandex.direct.intapi.webapp.semaphore.jvm.IntapiJvmSemaphoreInterceptor;
import ru.yandex.direct.web.core.WebLocaleResolver;
import ru.yandex.direct.web.core.security.WebLocaleResolverFilter;
import ru.yandex.direct.web.core.security.configuration.NetAclConfiguration;
import ru.yandex.direct.web.core.security.configuration.TvmConfiguration;
import ru.yandex.direct.web.core.security.netacl.NetAclInterceptor;
import ru.yandex.direct.web.core.security.tvm.TvmInterceptor;

@Configuration
@EnableWebMvc
@Import({
        IntapiConfiguration.class,
        SwaggerConfiguration.class,
        NetAclConfiguration.class,
        TvmConfiguration.class,
        IntapiLogConfiguration.class,
})
@ComponentScan(
        basePackages = {"ru.yandex.direct.intapi.webapp"}
)
public class WebAppConfiguration extends WebMvcConfigurerAdapter {

    private final NetAclInterceptor netAclInterceptor;
    private final IntapiSemaphoreInterceptor intapiSemaphoreInterceptor;
    private final MetricsInterceptor metricsInterceptor;
    private final TvmInterceptor tvmInterceptor;
    private final IntapiJvmSemaphoreInterceptor intapiJvmSemaphoreInterceptor;

    @Autowired
    public WebAppConfiguration(NetAclInterceptor netAclInterceptor,
                               IntapiSemaphoreInterceptor intapiSemaphoreInterceptor,
                               MetricsInterceptor metricsInterceptor,
                               TvmInterceptor tvmInterceptor,
                               IntapiJvmSemaphoreInterceptor intapiJvmSemaphoreInterceptor) {
        this.netAclInterceptor = netAclInterceptor;
        this.intapiSemaphoreInterceptor = intapiSemaphoreInterceptor;
        this.metricsInterceptor = metricsInterceptor;
        this.tvmInterceptor = tvmInterceptor;
        this.intapiJvmSemaphoreInterceptor = intapiJvmSemaphoreInterceptor;
    }

    @Bean(name = "intapiFilter")
    @Autowired
    public Filter intapiFilter(TraceContextFilter traceContextFilter, IntapiLoggingFilter intapiLoggingFilter,
                               MetricsFilter metricsFilter) {
        CompositeFilter compositeFilter = new CompositeFilter();
        // Порядок следования фильтров важен. traceContextFilter должен идти первым
        compositeFilter.setFilters(Arrays.asList(
                traceContextFilter,
                webLocaleResolverFilter(),
                intapiLoggingFilter,
                metricsFilter
        ));
        return compositeFilter;
    }

    @Bean
    public LocaleResolver localeResolver() {
        return new WebLocaleResolver();
    }

    @Bean
    public WebLocaleResolverFilter webLocaleResolverFilter() {
        return new WebLocaleResolverFilter();
    }

    @Bean
    public LoggingSettings loggingDefaults() {
        return new LoggingSettings(10 * 1024 * 1024, 10 * 1024);
    }

    @Bean
    public LoggingConfigurerInterceptor loggingConfigurerInterceptor() {
        return new LoggingConfigurerInterceptor(loggingDefaults());
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(loggingConfigurerInterceptor());
        registry.addInterceptor(metricsInterceptor);
        registry.addInterceptor(netAclInterceptor);
        registry.addInterceptor(tvmInterceptor);
        registry.addInterceptor(intapiSemaphoreInterceptor);
        registry.addInterceptor(intapiJvmSemaphoreInterceptor);
    }

    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        converters.removeIf(converter -> converter instanceof MappingJackson2XmlHttpMessageConverter);
        converters.add(new DirectTsvMessageConverter());
        converters.add(new ProtobufHttpMessageConverter());
        // модули JsonUtils.createLocalDateTimeModule() и JsonUtils.createLocalDateModule() приводят
        // к падению части тестов
    }
}
