package ru.yandex.direct.intapi.entity.absegment.controller;

import java.util.List;
import java.util.stream.Collectors;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.entity.absegment.model.AbSegmentModel;
import ru.yandex.direct.intapi.entity.absegment.model.AbSegmentsList;
import ru.yandex.direct.intapi.entity.absegment.service.AbSegmentService;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_PROD;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_TEST;
import static ru.yandex.direct.tvm.TvmService.METRIKA_INTERNAL_API_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_INTERNAL_API_TEST;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_PROD;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_TEST;

@Api(tags = "absegments")
@RestController
@RequestMapping(value = "absegments", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@AllowServices(production = {SURVEYS_PROD, METRIKA_INTERNAL_API_PROD, GEO_PRODUCT_PROD},
        testing = {DIRECT_DEVELOPER, SURVEYS_TEST, METRIKA_INTERNAL_API_TEST, GEO_PRODUCT_TEST})
public class AbSegmentController {
    private final AbSegmentService abSegmentService;

    @Autowired
    public AbSegmentController(AbSegmentService abSegmentService) {
        this.abSegmentService = abSegmentService;
    }

    @ApiOperation(value = "Get ab segments by campaign ids", httpMethod = "GET")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Ok", response = AbSegmentsList.class)
    )
    @RequestMapping(path = "/", method = RequestMethod.GET)
    public AbSegmentsList getAbSegments(@RequestParam(value = "campaignIds") List<Long> campaignIds) {
        return abSegmentService.getAbSegments(campaignIds);
    }

    @ApiOperation(value = "Save ab segments", httpMethod = "POST")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Ok", response = WebResponse.class)
    )
    @RequestMapping(path = "/save", method = RequestMethod.POST)
    public WebResponse saveAbSegments(@RequestBody AbSegmentsList request) {
        if (!validateListNotEmptyAndAllCampaignsIsDifferent(request)) {
            return new WebErrorResponse("Empty list or same campaignIds in request",
                    "Only different campaigns supports now");
        }
        abSegmentService.saveAbSegments(request);
        return new WebSuccessResponse();
    }

    @ApiOperation(value = "Unbind ab segments from campaigns", httpMethod = "POST")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Ok", response = WebResponse.class)
    )
    @RequestMapping(path = "/unbind", method = RequestMethod.POST)
    public WebResponse unbindAbSegments(@RequestParam(value = "campaignIds") List<Long> campaignIds) {
        abSegmentService.unbindAbSegments(campaignIds);
        return new WebSuccessResponse();
    }

    private static boolean validateListNotEmptyAndAllCampaignsIsDifferent(AbSegmentsList request) {
        if (request.getConditions() == null || request.getConditions().isEmpty()) {
            return false;
        }
        return request.getConditions().size() ==
                request.getConditions().stream().map(AbSegmentModel::getCampaignId).collect(Collectors.toSet()).size();
    }
}
