package ru.yandex.direct.intapi.entity.absegment.service;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.absegment.model.AbSegmentModel;
import ru.yandex.direct.intapi.entity.absegment.model.AbSegmentsList;

import static org.apache.commons.collections.CollectionUtils.isEmpty;

@Service
public class AbSegmentService {

    private static final long AB_SEGMENT_CORRECTION = 2_500_000_000L;

    private final RetargetingConditionRepository retargetingConditionRepository;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;

    public AbSegmentService(RetargetingConditionRepository retargetingConditionRepository,
                            CampaignRepository campaignRepository,
                            ShardHelper shardHelper) {
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
    }

    public AbSegmentsList getAbSegments(List<Long> campaignIds) {
        Map<Long, RetargetingCondition> conditions = new HashMap<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach((shard, campsForShard) -> {
            conditions.putAll(retargetingConditionRepository
                    .getRetConditionsByCampaignIds(shard, campsForShard, ConditionType.ab_segments));
        });

        List<AbSegmentModel> entries =
                EntryStream.of(conditions).mapToValue((k, v) -> AbSegmentService.convertModel(v, k)).values().toList();
        AbSegmentsList brandLiftResponse = new AbSegmentsList();
        brandLiftResponse.setConditions(entries);
        return brandLiftResponse;
    }

    public void saveAbSegments(AbSegmentsList response) {
        for (AbSegmentModel segment : response.getConditions()) {
            Long campaignId = segment.getCampaignId();
            int shard = shardHelper.getShardByCampaignId(campaignId);
            CampaignSimple campaign =
                    campaignRepository.getCampaignsSimple(shard, Collections.singletonList(campaignId)).get(campaignId);
            RetargetingCondition condition = convertSegment(campaign, segment);
            Long retargetingConditionId =
                    retargetingConditionRepository.add(shard, Collections.singletonList(condition)).get(0);
            campaignRepository.bindCampaignAndAbSegment(shard, campaignId, retargetingConditionId);
        }
    }

    public void unbindAbSegments(List<Long> campaignIds) {
        if (isEmpty(campaignIds)) {
            return;
        }
        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach(
                (shard, campsForShard) -> campaignRepository.unbindCampaignFromAbSegment(shard, campaignIds));
    }

    private static RetargetingCondition convertSegment(CampaignSimple campaign, AbSegmentModel segment) {
        Goal goal = new Goal();
        goal.withType(GoalType.AB_SEGMENT)
                .withId(segment.getSegmentId() + AB_SEGMENT_CORRECTION)
                .withTime(0);

        Rule rule = new Rule()
                .withSectionId(segment.getExperimentId())
                .withGoals(Collections.singletonList(goal))
                .withType(RuleType.OR);
        RetargetingCondition condition = new RetargetingCondition();
        condition.withRules(Collections.singletonList(rule))
                .withType(ConditionType.ab_segments)
                .withClientId(campaign.getClientId())
                .withName("ab segment condition")
                .withDescription("ab segment condition")
                .withDeleted(false)
                .withInterest(false)
                .withLastChangeTime(LocalDateTime.now());
        return condition;
    }

    private static AbSegmentModel convertModel(RetargetingCondition condition, Long campaignId) {
        AbSegmentModel abSegmentModel = new AbSegmentModel();
        abSegmentModel.setSegmentId(condition.getRules().get(0).getGoals().get(0).getId() - AB_SEGMENT_CORRECTION);
        abSegmentModel.setExperimentId(condition.getRules().get(0).getSectionId());
        abSegmentModel.setCampaignId(campaignId);
        return abSegmentModel;
    }
}
