package ru.yandex.direct.intapi.entity.adgroupadditionaltargeting.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.tvm.AllowServices;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_PROD;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_TEST;

@RestController
@Api(value = "Контроллер для управления расширенным таргетингом групп")
@RequestMapping(value = "/adgroups/targeting/additional")
@AllowServices(production = {GEO_PRODUCT_PROD},
        testing = {GEO_PRODUCT_TEST, DIRECT_DEVELOPER})
public class AdGroupAdditionalTargetingController {
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public AdGroupAdditionalTargetingController(
            AdGroupRepository adGroupRepository, ShardHelper shardHelper) {
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
    }
    @ApiOperation(
            httpMethod = "POST",
            value = "Задать на кампанию список yandex uid-ов которым разрешено показывать материалы этой группы",
            notes = "Если список yandex uid-ов не передан, то он сбрасывается,"
                    + " т.е. материалы группы можно будет неограниченному количествую пользователей"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 404, message = "Not found", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "OK")
            }
    )
    @RequestMapping(
            path = "yandexuids",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    public HttpStatus setAdGroupYandexUidList(
            @ApiParam(value = "Id группы", required = true)
            @RequestParam @Nonnull Long adGroupId,
            @ApiParam(value = "Список yandexuid-ов")
            @RequestBody(required = false) Set<String> yandexUids
    ) {
        int shard = shardHelper.getShardByGroupId(adGroupId);
        if (shard == NO_SHARD) {
            throw new IntApiException(
                    HttpStatus.NOT_FOUND,
                    new ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "Not found"));
        }

        Map<Long, ClientId> clientIdsByAdGroupIds = adGroupRepository.getClientIdsByAdGroupIds(shard,
                Collections.singleton(adGroupId));
        ClientId clientId = clientIdsByAdGroupIds.get(adGroupId);

        var adGroups = adGroupRepository.getAdGroups(shard, singleton(adGroupId));

        List<String> newYandexUids = yandexUids == null ? emptyList() : List.copyOf(yandexUids);
        ArrayList<AppliedChanges<AdGroup>> changes = new ArrayList<>();
        for (AdGroup ag : adGroups) {
            ModelChanges<AdGroup> mc = ModelChanges.build(ag, AdGroup.YANDEX_UIDS, newYandexUids);
            mc.process(StatusBsSynced.NO, AdGroup.STATUS_BS_SYNCED);
            changes.add(mc.applyTo(ag));
        }

        adGroupRepository.updateAdGroups(shard, clientId, changes);

        return HttpStatus.OK;
    }

}
