package ru.yandex.direct.intapi.entity.adgroupbstags.service;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupBsTags;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.model.PageGroupTagEnum;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupBsTagsService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.adgroupbstags.model.AdGroupBsTagsResponse;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.notFound;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRightsCantWrite;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class AdGroupBsTagsIntapiService {
    private static final AdGroupBsTags DEFAULT_YDO_ADGROUP_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.YDO_ADGROUP_BS_TAG))
            .withTargetTags(singletonList(TargetTagEnum.YDO_ADGROUP_BS_TAG));
    private static final Set<String> ALL_YDO_ADGROUP_TAGS = Set.of(
            TargetTagEnum.YDO_ADGROUP_BS_TAG.getTypedValue(), "yndx-services-exp-1", "yndx-services-exp-2",
            "yndx-services-exp-3", "yndx-services-exp-4", "yndx-services-exp-5");

    private static final Set<String> ALL_EDA_ADGROUP_TAGS = Set.of("eda", "eda_cpm");

    private static final AdGroupBsTags MAPS_TEST_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.MAPS_TEST))
            .withTargetTags(singletonList(TargetTagEnum.MAPS_TEST));
    private static final AdGroupBsTags TURBO_ECOM_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.TURBO_ECOM))
            .withTargetTags(singletonList(TargetTagEnum.TURBO_ECOM));
    private static final AdGroupBsTags EDADEAL_GEOPROD_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.EDADEAL_GEOPROD))
            .withTargetTags(singletonList(TargetTagEnum.EDADEAL_GEOPROD));
    private static final AdGroupBsTags MINI_APP_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.MINIAPP))
            .withTargetTags(singletonList(TargetTagEnum.MINIAPP));
    private static final AdGroupBsTags TURBO_AND_MINI_APP_TAG = new AdGroupBsTags()
            .withPageGroupTags(singletonList(PageGroupTagEnum.TURBO_AND_MINIAPP))
            .withTargetTags(singletonList(TargetTagEnum.TURBO_AND_MINIAPP));
    private final AdGroupBsTagsService adGroupBsTagsService;
    private final ValidationResultConversionService validationResultConversionService;
    private final RbacService rbacService;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;

    @Autowired
    public AdGroupBsTagsIntapiService(AdGroupBsTagsService adGroupBsTagsService,
                                      ValidationResultConversionService validationResultConversionService,
                                      RbacService rbacService,
                                      AdGroupRepository adGroupRepository,
                                      ShardHelper shardHelper,
                                      CampaignRepository campaignRepository) {
        this.adGroupBsTagsService = adGroupBsTagsService;
        this.validationResultConversionService = validationResultConversionService;
        this.rbacService = rbacService;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
    }

    public IntapiValidationResponse setYdoAdGroupBsTags(@Nonnull Long operatorUid, @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setYdoAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public IntapiValidationResponse setYdoAdGroupBsTags(@Nonnull List<Long> adGroupIds,
                                                        @Nonnull List<String> tags) {
        ValidationResult<Void, Defect> result = validateYdoAdGroupTypeAndTags(adGroupIds, tags);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        Set<String> uniqueTags = new HashSet<>(tags);
        adGroupBsTagsService.setRawTagsByAdgroupIds(uniqueTags, uniqueTags, adGroupIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public AdGroupBsTagsResponse getYdoAdGroupBsTags(@Nonnull List<Long> adGroupIds) {
        ValidationResult<Void, Defect> result = validateYdoAdGroupTypeAndTags(adGroupIds, null);
        if (result.hasAnyErrors()) {
            IntapiValidationResult intapiValidationResult =
                    validationResultConversionService.buildIntapiValidationResult(result);
            return new AdGroupBsTagsResponse(intapiValidationResult);
        }
        return new AdGroupBsTagsResponse(new IntapiValidationResult(),
                adGroupBsTagsService.getTargetTagsByAdgroupIds(adGroupIds));
    }

    public IntapiValidationResponse setEdaAdGroupBsTags(@Nonnull List<Long> adGroupIds,
                                                        @Nonnull List<String> tags) {
        ValidationResult<Void, Defect> result = validateEdaAdGroupTags(adGroupIds, tags);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        Set<String> uniqueTags = new HashSet<>(tags);
        adGroupBsTagsService.setRawTagsByAdgroupIds(uniqueTags, uniqueTags, adGroupIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public AdGroupBsTagsResponse getEdaAdGroupBsTags(@Nonnull List<Long> adGroupIds) {
        ValidationResult<Void, Defect> result = validateEdaAdGroupTags(adGroupIds, null);
        if (result.hasAnyErrors()) {
            IntapiValidationResult intapiValidationResult =
                    validationResultConversionService.buildIntapiValidationResult(result);
            return new AdGroupBsTagsResponse(intapiValidationResult);
        }
        return new AdGroupBsTagsResponse(new IntapiValidationResult(),
                adGroupBsTagsService.getTargetTagsByAdgroupIds(adGroupIds));
    }

    @QueryWithForbiddenShardMapping("pid")
    private ValidationResult<Void, Defect> validateYdoAdGroupTypeAndTags(List<Long> adGroupIds,
                                                                         List<String> tags) {
        Map<Long, ContentPromotionAdgroupType> typesByIds = new HashMap<>();
        shardHelper.groupByShard(adGroupIds, ShardKey.PID)
                .chunkedBy(1000)
                .forEach((shard, pids) ->
                        typesByIds.putAll(adGroupRepository.getContentPromotionAdGroupTypesByIds(shard, pids))
                );
        ItemValidationBuilder<Void, Defect> vb = ItemValidationBuilder.of(null);
        vb.list(adGroupIds, "ad_group_ids")
                .checkEach(fromPredicate(typesByIds::containsKey, notFound()))
                .checkEach(fromPredicate(pid -> typesByIds.get(pid) == ContentPromotionAdgroupType.SERVICE,
                        adGroupTypeNotSupported()), When.isValid());
        vb.list(tags, "bs_tags")
                .check(notEmptyCollection(), When.notNull())
                .checkEach(notNull())
                .checkEach(inSet(ALL_YDO_ADGROUP_TAGS));
        return vb.getResult();
    }

    @QueryWithForbiddenShardMapping("pid")
    private ValidationResult<Void, Defect> validateEdaAdGroupTags(List<Long> adGroupIds,
                                                                         List<String> tags) {
        ItemValidationBuilder<Void, Defect> vb = ItemValidationBuilder.of(null);
        var validAdGroupIds = new HashSet<Long>();
        shardHelper.groupByShard(adGroupIds, ShardKey.PID)
                .chunkedBy(1000)
                .forEach((shard, pids) ->
                        validAdGroupIds.addAll(
                                EntryStream.of(campaignRepository.getCampaignsSimpleByAdGroupIds(shard, pids))
                                        .filterValues(campaign -> campaign.getSource() == CampaignSource.EDA)
                                        .keys()
                                        .toSet())
                );

        vb.list(adGroupIds, "ad_group_ids")
                .check(fromPredicate(validAdGroupIds::containsAll, notFound()));

        vb.list(tags, "bs_tags")
                .check(notEmptyCollection(), When.notNull())
                .checkEach(notNull())
                .checkEach(inSet(ALL_EDA_ADGROUP_TAGS));
        return vb.getResult();
    }

    public IntapiValidationResponse setMapsTestAdGroupBsTags(@Nonnull Long operatorUid,
                                                             @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setMapsTestAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public IntapiValidationResponse setTurboEcomAdGroupBsTags(@Nonnull Long operatorUid,
                                                              @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setTurboEcomAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public IntapiValidationResponse setEdadealGeoprodAdGroupBsTags(@Nonnull Long operatorUid,
                                                              @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setEdadealGeoprodAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public IntapiValidationResponse setMiniAppAdGroupBsTags(@Nonnull Long operatorUid,
                                                            @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setMiniAppAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    public IntapiValidationResponse setTurboAndMiniAppEcomAdGroupBsTags(@Nonnull Long operatorUid,
                                                                        @Nonnull List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> result = validateCampaignIdsAccess(operatorUid, campaignIds);
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        setTurboAndMiniAppEcomAdGroupBsTagsInternal(operatorUid, campaignIds);
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    private ValidationResult<List<Long>, Defect> validateCampaignIdsAccess(
            Long operatorUid, List<Long> campaignIds) {
        Set<Long> operatorAvailableCampaigns = rbacService.getWritableCampaigns(operatorUid, campaignIds);
        return ListValidationBuilder.<Long, Defect>of(campaignIds)
                .checkEach(fromPredicate(cid -> operatorAvailableCampaigns.contains(cid), noRightsCantWrite()))
                .getResult();
    }

    private void setYdoAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> DEFAULT_YDO_ADGROUP_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }

    private void setMapsTestAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> MAPS_TEST_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }

    private void setTurboEcomAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> TURBO_ECOM_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }

    private void setEdadealGeoprodAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> EDADEAL_GEOPROD_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }

    private void setMiniAppAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> MINI_APP_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }

    private void setTurboAndMiniAppEcomAdGroupBsTagsInternal(long operatorUid, @Nonnull List<Long> campaignIds) {
        Map<Long, AdGroupBsTags> tagsByCampaignIds = campaignIds.stream().collect(
                Collectors.toMap(identity(), t -> TURBO_AND_MINI_APP_TAG, (tag1, tag2) -> tag1));
        adGroupBsTagsService.setAdGroupBsTagsByCampaignIds(operatorUid, tagsByCampaignIds);
    }
}
