package ru.yandex.direct.intapi.entity.auction.model;

import java.math.RoundingMode;
import java.util.List;

import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.MicroMoneyConverter;
import ru.yandex.direct.currency.MoneyUtils;

import static ru.yandex.direct.core.entity.bids.container.interpolator.PointConverter.TRAFFIC_VOLUME_SCALE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class AuctionValuesConverter {
    private AuctionValuesConverter() {
    }

    public static List<TrafaretBidItem> toTrafaretBidItems(List<AuctionItem> auctionItems, CurrencyCode currencyCode) {
        return mapList(auctionItems, auctionItem -> toTrafaretBidItem(auctionItem, currencyCode));
    }

    private static TrafaretBidItem toTrafaretBidItem(AuctionItem auctionItem, CurrencyCode currencyCode) {
        MicroMoneyConverter moneyConverter = MicroMoneyConverter.roundingUp(currencyCode);
        return new TrafaretBidItem()
                .withBid(moneyConverter.apply(auctionItem.getBid()))
                .withPositionCtrCorrection(auctionItem.getTrafficVolume())
                .withPrice(moneyConverter.apply(auctionItem.getAmnestyPrice()));
    }

    public static List<AuctionItem> toAuctionItems(List<TrafaretBidItem> trafaretBidItems) {
        return mapList(trafaretBidItems, AuctionValuesConverter::toAuctionItem);
    }

    private static AuctionItem toAuctionItem(TrafaretBidItem trafaretBidItem) {
        return new AuctionItem()
                .withAmnestyPrice(trafaretBidItem.getPrice().micros())
                .withBid(trafaretBidItem.getBid().micros())
                .withTrafficVolume(trafaretBidItem.getPositionCtrCorrection());
    }

    public static List<Double> convertCalculatingTrafficVolumes(List<Long> calculatingTrafficVolumes) {
        return mapList(calculatingTrafficVolumes, trafficVolume -> trafficVolume.doubleValue() / TRAFFIC_VOLUME_SCALE);
    }

    public static List<AuctionItem> roundAuctionItemsValues(List<AuctionItem> auctionItems, CurrencyCode currencyCode) {
        return mapList(auctionItems, x -> roundAuctionItemValues(x, currencyCode));
    }

    private static AuctionItem roundAuctionItemValues(AuctionItem auctionItem, CurrencyCode currencyCode) {
        long roundedAmnestyPrice = MoneyUtils.roundMicrosToAuctionStep(
                auctionItem.getAmnestyPrice(), currencyCode.getCurrency(), RoundingMode.UP);
        long roundedBid = MoneyUtils.roundMicrosToAuctionStep(
                auctionItem.getBid(), currencyCode.getCurrency(), RoundingMode.UP);

        double trafficVolume = ((double) auctionItem.getTrafficVolume()) / TRAFFIC_VOLUME_SCALE;
        long roundedTrafficVolume = Math.round(trafficVolume) * TRAFFIC_VOLUME_SCALE;
        return new AuctionItem()
                .withAmnestyPrice(roundedAmnestyPrice)
                .withBid(roundedBid)
                .withTrafficVolume(roundedTrafficVolume);
    }
}
