package ru.yandex.direct.intapi.entity.auction.service;

import java.util.ArrayList;
import java.util.List;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.bids.container.interpolator.CapKey;
import ru.yandex.direct.core.entity.bids.interpolator.InterpolatorService;
import ru.yandex.direct.intapi.entity.auction.model.AuctionItem;
import ru.yandex.direct.intapi.entity.auction.model.InterpolateRequest;
import ru.yandex.direct.intapi.entity.auction.model.PhraseAuctionItem;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.entity.auction.model.AuctionValuesConverter.convertCalculatingTrafficVolumes;
import static ru.yandex.direct.intapi.entity.auction.model.AuctionValuesConverter.roundAuctionItemsValues;
import static ru.yandex.direct.intapi.entity.auction.model.AuctionValuesConverter.toAuctionItems;
import static ru.yandex.direct.intapi.entity.auction.model.AuctionValuesConverter.toTrafaretBidItems;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiDefects.notEmptyRequest;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;


@Service
public class InterpolateAuctionItemsService {
    private static final Logger logger = LoggerFactory.getLogger(InterpolateAuctionItemsService.class);
    // в зависимости от того, какой максимальный TRAFFIC_VOLUME,
    // при сглаживании будет применяться соответсвующий колпак.
    // на текущий момент есть разделение на два случая: когда максимум < 500000 и когда больше.
    private static final int HALF_OF_THE_DEFAULT_MAX_TRAFFIC_VOLUME = 500000;
    private final InterpolatorService interpolatorService;

    @Autowired
    public InterpolateAuctionItemsService(InterpolatorService interpolatorService) {
        this.interpolatorService = interpolatorService;
    }

    public List<PhraseAuctionItem> interpolate(InterpolateRequest interpolateRequest) {
        ValidationResult<List<PhraseAuctionItem>, IntApiDefect> validationResult =
                validatePhraseAuctionItems(interpolateRequest.getPhraseBidsItemList());
        checkResult(validationResult);

        List<PhraseAuctionItem> results = new ArrayList<>(interpolateRequest.getPhraseBidsItemList().size());

        for (PhraseAuctionItem phraseBidsItem : interpolateRequest.getPhraseBidsItemList()) {
            boolean isTop = StreamEx.of(phraseBidsItem.getAuctionItems())
                    .map(AuctionItem::getTrafficVolume)
                    .anyMatch(x -> x > HALF_OF_THE_DEFAULT_MAX_TRAFFIC_VOLUME);

            CapKey capKey = new CapKey(phraseBidsItem.getIsExactMatch(), isTop ? 1 : 0, phraseBidsItem.getDomain());
            try {
                List<TrafaretBidItem> trafaretBidItems = toTrafaretBidItems(phraseBidsItem.getAuctionItems(),
                        phraseBidsItem.getCurrencyCode());

                List<Double> convertedCalculatingTrafficVolumes =
                        convertCalculatingTrafficVolumes(phraseBidsItem.getCalculatingTrafficVolumes());

                List<TrafaretBidItem> interpolatedTrafaretBidItems = interpolatorService
                        .getInterpolatedTrafaretBidItems(capKey, trafaretBidItems, convertedCalculatingTrafficVolumes,
                                phraseBidsItem.getCurrencyCode());

                results.add(new PhraseAuctionItem().withAuctionItems(toAuctionItems(interpolatedTrafaretBidItems)));
            } catch (Exception e) {
                logger.warn("Can't interpolate points", e);
                results.add(new PhraseAuctionItem()
                        .withAuctionItems(roundAuctionItemsValues(phraseBidsItem.getAuctionItems(),
                                phraseBidsItem.getCurrencyCode())));
            }

        }

        return results;
    }

    public ValidationResult<List<PhraseAuctionItem>, IntApiDefect> validatePhraseAuctionItems(
            List<PhraseAuctionItem> phraseAuctionItems) {
        ListValidationBuilder<PhraseAuctionItem, IntApiDefect> v =
                ListValidationBuilder.of(phraseAuctionItems, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.checkEachBy(this::validatePhraseAuctionItem, When.notNull());
        }

        return v.getResult();
    }

    private ValidationResult<PhraseAuctionItem, IntApiDefect> validatePhraseAuctionItem(
            PhraseAuctionItem phraseAuctionItem) {
        ItemValidationBuilder<PhraseAuctionItem, IntApiDefect> v =
                ItemValidationBuilder.of(phraseAuctionItem, IntApiDefect.class);
        v.item(phraseAuctionItem.getAuctionItems(), PhraseAuctionItem.AUCTION_ITEMS)
                .check(notNull());
        v.item(phraseAuctionItem.getCurrencyCode(), PhraseAuctionItem.CURRENCY_CODE)
                .check(notNull());
        return v.getResult();
    }
}
