package ru.yandex.direct.intapi.entity.balanceclient.container;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;

import ru.yandex.direct.intapi.entity.balanceclient.model.BalanceClientResult;

/**
 * Класс-обертка для представления ответа Балансу.
 * <p>
 * Как баланс обрабатывает полученные ответы:
 * Если код ответа {@link HttpStatus#OK} - содержимое ответа игнорируется, нотификация удаляется из очерди нотификатора
 * При других кодах (подразумевающих ошибку, например {@link HttpStatus#BAD_REQUEST}
 * или {@link HttpStatus#INTERNAL_SERVER_ERROR}) - нотификация останется в очереди, содержимое ответа будет залогировано
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-62594">переход XMLRPC->JSON-REST</a>
 */
@ParametersAreNonnullByDefault
public class BalanceClientResponse extends ResponseEntity<BalanceClientResult> {
    public BalanceClientResponse(BalanceClientResult body, HttpStatus status) {
        super(body, defaultHeaders(), status);
    }

    public static BalanceClientResponse success(String message) {
        return new BalanceClientResponse(new BalanceClientResult(BalanceClientResult.OK_CODE, message), HttpStatus.OK);
    }

    public static BalanceClientResponse success() {
        return success("");
    }

    public static BalanceClientResponse error(int code, String message) {
        return new BalanceClientResponse(new BalanceClientResult(code, message), HttpStatus.BAD_REQUEST);
    }

    public static BalanceClientResponse criticalError(String message) {
        return new BalanceClientResponse(new BalanceClientResult(BalanceClientResult.INTERNAL_ERROR_CODE, message),
                HttpStatus.INTERNAL_SERVER_ERROR);
    }

    public static BalanceClientResponse errorConflict(int code, String message) {
        return new BalanceClientResponse(new BalanceClientResult(code, message), HttpStatus.CONFLICT);
    }

    /**
     * Ошибка с HTTP статусом {@code 423 Locked}
     */
    public static BalanceClientResponse errorLocked(int code, String message) {
        return new BalanceClientResponse(new BalanceClientResult(code, message), HttpStatus.LOCKED);
    }

    private static HttpHeaders defaultHeaders() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        return headers;
    }
}
