package ru.yandex.direct.intapi.entity.balanceclient.controller;

import java.util.List;

import javax.annotation.Nonnull;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import ru.yandex.direct.core.entity.balanceaggrmigration.lock.AggregateMigrationRedisLockService;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientGetHostingsResponse;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.container.LockCampaignRequest;
import ru.yandex.direct.intapi.entity.balanceclient.model.BalanceClientResult;
import ru.yandex.direct.intapi.entity.balanceclient.model.GetHostingsResult;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyAgencyAdditionalCurrenciesParameters;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientParameters;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyPromocodeParameters;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyAgencyAdditionalCurrenciesService;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyClientCashBackService;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyClientService;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyOrderService;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyPromoOrderService;
import ru.yandex.direct.intapi.entity.balanceclient.service.NotifyPromocodeService;
import ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.tvm.AllowServices;

import static java.lang.String.format;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.INVALID_SERVICE_ID_ERROR_CODE;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.INVALID_SERVICE_MESSAGE;
import static ru.yandex.direct.tvm.TvmService.BALANCE_PROD;
import static ru.yandex.direct.tvm.TvmService.BALANCE_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.tvm.TvmService.YABS_BANANA_PROD;
import static ru.yandex.direct.tvm.TvmService.YABS_BANANA_TEST;

@Api
@Controller
@RequestMapping(value = "BalanceClient", produces = MediaType.APPLICATION_JSON_VALUE)
public class BalanceClientController {

    static final BalanceClientResponse INVALID_REQUEST_SIZE =
            BalanceClientResponse.criticalError("array must contain 1 element");

    private final NotifyClientService notifyClientService;
    private final NotifyOrderService notifyOrderService;
    private final NotifyPromoOrderService notifyPromoOrderService;
    private final NotifyPromocodeService notifyPromocodeService;
    private final NotifyAgencyAdditionalCurrenciesService notifyAgencyAdditionalCurrenciesService;
    private final HostingsHandler hostingsHandler;
    private final Integer yandexAgencyServiceId;
    private final AggregateMigrationRedisLockService migrationRedisLockService;
    private final NotifyClientCashBackService notifyClientCashBackService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public BalanceClientController(
            NotifyClientService notifyClientService,
            NotifyOrderService notifyOrderService, NotifyPromoOrderService notifyPromoOrderService,
            NotifyAgencyAdditionalCurrenciesService notifyAgencyAdditionalCurrenciesService,
            NotifyPromocodeService notifyPromocodeService,
            HostingsHandler hostingsHandler,
            @Value("${balance.yandexAgencyServiceId}") int yandexAgencyServiceId,
            AggregateMigrationRedisLockService migrationRedisLockService,
            NotifyClientCashBackService notifyClientCashBackService
    ) {
        this.notifyClientService = notifyClientService;
        this.notifyOrderService = notifyOrderService;
        this.notifyPromoOrderService = notifyPromoOrderService;
        this.notifyAgencyAdditionalCurrenciesService = notifyAgencyAdditionalCurrenciesService;
        this.notifyPromocodeService = notifyPromocodeService;
        this.hostingsHandler = hostingsHandler;
        this.yandexAgencyServiceId = yandexAgencyServiceId;
        this.migrationRedisLockService = migrationRedisLockService;
        this.notifyClientCashBackService = notifyClientCashBackService;
    }

    BalanceClientController(NotifyOrderService notifyOrderService, int yandexAgencyServiceId) {
        this(null, notifyOrderService, null, null, null, null, yandexAgencyServiceId, null, null);
    }

    /**
     * Откуда цифра 2 в NotifyClient2?
     * В perl-Директе существовала ручка NotifyClient с позиционными (не именованными) аргументам.
     * NotifyClient2 её заменила.
     * <p>
     * Поскольку в java-Директе предыдущей версии метода никогда не существовало - цифра 2 по коду - опущена
     */
    @ApiOperation(value = "Нотификация об изменениях в свойствах клиента "
            + "(овердрафт, нерезидентство, компенсация минусов по ОС)",
            response = BalanceClientResult.class)
    @RequestMapping(value = "NotifyClient2", method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD},
            testing = {BALANCE_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public BalanceClientResponse notifyClient(
            @ApiParam(value = "параметры клиента", required = true)
            @RequestBody @Nonnull List<NotifyClientParameters> updateRequestList) {
        if (updateRequestList.size() != 1) {
            return INVALID_REQUEST_SIZE;
        }

        return notifyClientService.notifyClient(updateRequestList.get(0));
    }

    @ApiOperation(value = "Ping для BalanceClient-а. Нужно для тестовых вызовов в проде",
            response = BalanceClientResult.class)
    @RequestMapping(value = "Ping", method = {RequestMethod.GET, RequestMethod.POST})
    @AllowServices(production = {BALANCE_PROD}, testing = {BALANCE_TEST, DIRECT_DEVELOPER})
    public BalanceClientResponse ping() {
        return BalanceClientResponse.success();
    }

    @ApiOperation(value = "Отдает списки публичных доменов второго уровня и известных хостинг-площадок",
            response = GetHostingsResult.class)
    @RequestMapping(value = "GetHostings", method = {RequestMethod.GET})
    @AllowServices(production = {BALANCE_PROD}, testing = {BALANCE_TEST, DIRECT_DEVELOPER})
    public BalanceClientGetHostingsResponse getHostings() {
        return BalanceClientGetHostingsResponse
                .success(hostingsHandler.getHostings(), hostingsHandler.getPublicSecondLevelDomains());
    }

    /**
     * Откуда цифра 2 в NotifyOrder2?
     * В perl-Директе существовала ручка NotifyOrder с позиционными (не именованными) аргументам.
     * NotifyOrder2 её заменила.
     * <p>
     * Поскольку в java-Директе предыдущей версии метода никогда не существовало - цифра 2 по коду - опущена
     */
    @ApiOperation(value = "Принимает данные об изменении суммы на кампании от Баланса",
            response = BalanceClientResult.class)
    @RequestMapping(value = "NotifyOrder2", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD},
            testing = {BALANCE_TEST, DIRECT_WEB_TEST, DIRECT_INTAPI_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS},
            sandbox = {DIRECT_INTAPI_SANDBOX, DIRECT_API_SANDBOX},
            sandboxTesting = {DIRECT_INTAPI_SANDBOX_TEST, DIRECT_API_SANDBOX_TEST, DIRECT_DEVELOPER}
    )
    public BalanceClientResponse notifyOrder(
            @ApiParam(value = "параметры заказа", required = true)
            @RequestBody @Nonnull List<NotifyOrderParameters> updateRequestList) {
        //https://st.yandex-team.ru/BALANCE-25695 баланс только NotifyOrder2 отправляет
        //"Принимает данные об изменении статуса платежа заказа у яндекс.агенств" если service_id = 177
        boolean isNotifyPromoOrderType = updateRequestList.stream()
                .anyMatch(order -> yandexAgencyServiceId.equals(order.getServiceId()));
        if (isNotifyPromoOrderType) {
            return notifyPromoOrderService.notifyPromoOrder(updateRequestList);
        }

        if (updateRequestList.size() != 1) {
            return INVALID_REQUEST_SIZE;
        }
        return notifyOrderService.notifyOrder(updateRequestList.get(0));
    }

    /**
     * метод дублирует notifyOrder, но у него другой ACL и проверки внутри:
     * к нему есть доступ у Бананы (banana.yandex-team.ru), так что через этот метод
     * можно передавать уведомления только по банановому serviceId = 67
     * (а {@link NotifyOrderValidationService} гарантирует, что с таким serviceId
     * зачислять деньги можно только на кампании с типом internal_autobudget)
     */
    @ApiOperation(value = "Принимает данные об изменении суммы на кампании от Бананы",
            response = BalanceClientResult.class)
    @RequestMapping(value = "NotifyOrderBanana", method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD, YABS_BANANA_PROD},
            testing = {BALANCE_TEST, YABS_BANANA_TEST, DIRECT_WEB_TEST, DIRECT_INTAPI_TEST,
                    DIRECT_DEVELOPER, DIRECT_AUTOTESTS}
    )
    public BalanceClientResponse notifyOrderBanana(
            @ApiParam(value = "параметры заказа", required = true)
            @RequestBody @Nonnull List<NotifyOrderParameters> updateRequestList) {
        for (NotifyOrderParameters request : updateRequestList) {
            if (request.getServiceId() != 67) {
                return BalanceClientResponse.error(INVALID_SERVICE_ID_ERROR_CODE,
                        format(INVALID_SERVICE_MESSAGE, request.getServiceId()));
            }
        }

        return notifyOrder(updateRequestList);
    }

    /**
     * Получаем списки дополнительных валют для агентств (список валют, на которые у агентства подписано
     * доп.соглашение) и сроков их действия.
     * Доп. валюты используются при создании клиентов агенства.
     * Описание см. в BALANCE-11907.
     */
    @ApiOperation(
            value = "Получает списки дополнительных валют для агентств (список валют, на которые у агентства " +
                    "подписано доп.соглашение) и сроков их действия.",
            httpMethod = "POST",
            nickname = "notifyAgencyAdditionalCurrencies"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = BalanceClientResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = BalanceClientResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = BalanceClientResponse.class)
            }
    )
    @RequestMapping(value = "NotifyAgencyAdditionalCurrencies", method = RequestMethod.POST, consumes =
            MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD},
            testing = {BALANCE_TEST, DIRECT_WEB_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public BalanceClientResponse notifyAgencyAdditionalCurrencies(
            @ApiParam(value = "параметры валют агентства", required = true)
            @RequestBody @Nonnull List<NotifyAgencyAdditionalCurrenciesParameters> agencyCurrenciesRequestList) {
        if (agencyCurrenciesRequestList.size() != 1) {
            return INVALID_REQUEST_SIZE;
        }
        return notifyAgencyAdditionalCurrenciesService.processAgency(agencyCurrenciesRequestList.get(0));
    }

    /**
     * Обработка списка действующих промокодов на заказе.
     *
     * @implNote может не работать в sandbox-окружении, так как у нас не реализована балансовая руча по отрыву промокода
     */
    @ApiOperation(value = "Принимает данные об изменении промкодных средств на заказе", response =
            BalanceClientResult.class)
    @PostMapping(value = "NotifyPromocode", consumes = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD}, testing = {BALANCE_TEST, DIRECT_DEVELOPER})
    public BalanceClientResponse notifyPromocode(
            @ApiParam(value = "идентификаторы заказа и список промокодов", required = true)
            @RequestBody @Nonnull List<NotifyPromocodeParameters> updateRequestList
    ) {
        if (updateRequestList.size() != 1) {
            return INVALID_REQUEST_SIZE;
        }

        return notifyPromocodeService.notifyPromocode(updateRequestList.get(0));
    }

    /**
     * Дополнительная ручка для взятия или освобождения лока на кампанию.
     * Для удобного интеграционного тестирования схем нотификаций.
     *
     * @param lockCampaignRequest контейнер флажка включения/снятия лока и номера общего счёта.
     * @return true/false - успешность запрошенной операции редислока на общий счёт
     */
    @ApiOperation(
            value = "Взятие лока на указанную кампанию (для тестов)",
            httpMethod = "POST",
            nickname = "redislock"
    )
    @RequestMapping(path = "redislock",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @AllowServices(production = {}, testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public ResponseEntity<Boolean> redisLock(
            @ApiParam(value = "ручка лока кампании", required = true)
            @RequestBody LockCampaignRequest lockCampaignRequest) {
        Long walletId = lockCampaignRequest.getWalletId();
        Boolean toLock = lockCampaignRequest.getLockFlag();
        if (toLock == null) {
            return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
        }

        boolean isSuccess;
        if (toLock) {
            DistributedLock lock = migrationRedisLockService.lock(walletId);
            isSuccess = lock.isLocked();
        } else {
            DistributedLock lock = migrationRedisLockService.buildLock(walletId);
            isSuccess = migrationRedisLockService.unlock(lock);
        }

        return new ResponseEntity<>(isSuccess, HttpStatus.OK);
    }

    /**
     * Прием нотификаций об изменении значения кэшбека у клиента.
     */
    @ApiOperation(value = "Принимает данные об изменении значения кэшбека у клиента", response =
            BalanceClientResult.class)
    @PostMapping(value = "NotifyClientCashback", consumes = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {BALANCE_PROD}, testing = {BALANCE_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public BalanceClientResponse notifyClientCashBack(
            @ApiParam(value = "Данные о кэшбеках клиентов", required = true)
            @RequestBody @Nonnull List<NotifyClientCashBackParameters> updateRequestList
    ) {
        if (updateRequestList.size() != 1) {
            return INVALID_REQUEST_SIZE;
        }

        return notifyClientCashBackService.notifyClientCashBack(updateRequestList.get(0));
    }
}
