package ru.yandex.direct.intapi.entity.balanceclient.model;

import java.math.BigDecimal;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

/**
 * Модель параметров запроса NotifyOrder2
 *
 * @see <a href=https://wiki.yandex-team.ru/balance/interfaces/notifications/#balanceclient.notifyorder2>
 * описание на вики</a>
 */
@ApiModel
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY,
        getterVisibility = JsonAutoDetect.Visibility.NONE,
        setterVisibility = JsonAutoDetect.Visibility.NONE,
        creatorVisibility = JsonAutoDetect.Visibility.NONE
)
public class NotifyOrderParameters {
    public static final String SERVICE_ID_FIELD_NAME = "ServiceID";
    public static final String CAMPAIGN_ID_FIELD_NAME = "ServiceOrderID";
    public static final String SUM_UNITS_FIELD_NAME = "ConsumeQty";
    public static final String CHIPS_COST_FIELD_NAME = "CompletionFixedMoneyQty";
    public static final String CHIPS_SPENT_FIELD_NAME = "CompletionFixedQty";
    public static final String SUM_REAL_MONEY_FIELD_NAME = "ConsumeMoneyQty";
    public static final String TOTAL_SUM_FIELD_NAME = "TotalConsumeQty";
    public static final String TID_FIELD_NAME = "Tid";
    public static final String CASHBACK_FIELD_NAME = "CashbackCurrentQty";
    public static final String TOTAL_CASHBACK_FIELD_NAME = "TotalCashbackCurrentQty";

    /**
     * Код сервиса
     */
    @JsonProperty(value = SERVICE_ID_FIELD_NAME)
    @ApiModelProperty(value = "Идентификатор сервиса", required = true, example = "7")
    private Integer serviceId;

    /**
     * Номер кампании в директе
     */
    @JsonProperty(value = CAMPAIGN_ID_FIELD_NAME, required = true)
    @ApiModelProperty(value = "Идентификатор кампании (заказа) на сервисе", required = true, example = "263")
    private Long campaignId;

    /**
     * Идентификатор транзакции (campaigns.balance_tid, должен все время увеличиваться)
     */
    @JsonProperty(value = TID_FIELD_NAME)
    @ApiModelProperty(value = "идентификатор транзакции, должен увеличиваться",
            required = true, example = "8861210358693")
    private Long tid;

    /**
     * Положенное на заказ кол-во единиц товара (для кампаний с продуктом в фишках это сумма в у.е.)
     */
    @JsonProperty(value = SUM_UNITS_FIELD_NAME)
    private BigDecimal sumUnits;

    /**
     * 1, если оплата была сертификатом (BALANCE-11401), которая используется для оплаты заказов с внутренней рекламой
     */
    @ApiModelProperty(value = "логический признак 'была ли оплата сертификатом' (BALANCE-11401)", example = "1")
    @JsonProperty(value = "Certificate")
    private Integer paidByCertificate;

    /**
     * Cумма денег с учётом бонуса от промокода
     */
    @JsonProperty(value = SUM_REAL_MONEY_FIELD_NAME)
    private BigDecimal sumRealMoney;

    /**
     * Валюта продукта в заказе
     */
    @JsonProperty(value = "ProductCurrency")
    private String productCurrency;

    /**
     * Стоимость в рублях истраченных на кампании фишек до её конвертации в рубли без копирования (BALANCE-17515)
     */
    @JsonProperty(value = CHIPS_COST_FIELD_NAME)
    private BigDecimal chipsCost;

    /**
     * Количество истраченных на кампании фишек до её конвертации в рубли без копирования (BALANCE-19017)
     */
    @JsonProperty(value = CHIPS_SPENT_FIELD_NAME)
    private BigDecimal chipsSpent;

    /**
     * Для кошельков и кампаний под кошельками, суммарно положенная на кошелек
     * + кампании под кошельком сумма в валюте кошелька
     * <p>
     * При конвертации без копирования YND_FIXED -> RUB умножаем на 30,
     * а затем уточненная сумма прийдет очередной нотификацией)
     */
    @JsonProperty(value = TOTAL_SUM_FIELD_NAME)
    private BigDecimal totalSum;

    /**
     * Кешбек кампании
     */
    @JsonProperty(value = CASHBACK_FIELD_NAME)
    private BigDecimal cashback;

    /**
     * Сумма кешбеков под кошельком
     */
    @JsonProperty(value = TOTAL_CASHBACK_FIELD_NAME)
    private BigDecimal totalCashback;

    public Integer getServiceId() {
        return serviceId;
    }

    public Long getCampaignId() {
        return campaignId;
    }

    public Long getTid() {
        return tid;
    }

    public BigDecimal getSumUnits() {
        return sumUnits;
    }

    public Integer getPaidByCertificate() {
        return paidByCertificate;
    }

    public BigDecimal getSumRealMoney() {
        return sumRealMoney;
    }

    public String getProductCurrency() {
        return productCurrency;
    }

    public BigDecimal getChipsCost() {
        return chipsCost;
    }

    public BigDecimal getChipsSpent() {
        return chipsSpent;
    }

    public BigDecimal getTotalSum() {
        return totalSum;
    }

    public BigDecimal getCashback() {
        return cashback;
    }

    public BigDecimal getTotalCashback() {
        return totalCashback;
    }

    public NotifyOrderParameters withServiceId(Integer serviceId) {
        this.serviceId = serviceId;
        return this;
    }

    public NotifyOrderParameters withCampaignId(Long campaignId) {
        this.campaignId = campaignId;
        return this;
    }

    public NotifyOrderParameters withTid(Long tid) {
        this.tid = tid;
        return this;
    }

    public NotifyOrderParameters withSumUnits(BigDecimal sumUnits) {
        this.sumUnits = sumUnits;
        return this;
    }

    public NotifyOrderParameters withPaidByCertificate(Integer paidByCertificate) {
        this.paidByCertificate = paidByCertificate;
        return this;
    }

    public NotifyOrderParameters withSumRealMoney(BigDecimal sumRealMoney) {
        this.sumRealMoney = sumRealMoney;
        return this;
    }

    public NotifyOrderParameters withProductCurrency(String productCurrency) {
        this.productCurrency = productCurrency;
        return this;
    }

    public NotifyOrderParameters withChipsCost(BigDecimal chipsCost) {
        this.chipsCost = chipsCost;
        return this;
    }

    public NotifyOrderParameters withChipsSpent(BigDecimal chipsSpent) {
        this.chipsSpent = chipsSpent;
        return this;
    }

    public NotifyOrderParameters withTotalSum(BigDecimal totalSum) {
        this.totalSum = totalSum;
        return this;
    }

    public NotifyOrderParameters withCashback(BigDecimal cashback) {
        this.cashback = cashback;
        return this;
    }

    public NotifyOrderParameters withTotalCashback(BigDecimal totalCashback) {
        this.totalCashback = totalCashback;
        return this;
    }

    /**
     * Проверяет "есть ли в нотификации деньги" (хотя бы одно денежное поле определено и не равно нулю).
     * Список проверяемых полей:
     * <ul>
     * <li>{@link #SUM_UNITS_FIELD_NAME}</li>
     * <li>{@link #SUM_REAL_MONEY_FIELD_NAME}</li>
     * <li>{@link #CHIPS_COST_FIELD_NAME}</li>
     * <li>{@link #CHIPS_SPENT_FIELD_NAME}</li>
     * <li>{@link #TOTAL_SUM_FIELD_NAME}</li>
     * </ul>
     *
     * @return {@code true}, если хотя бы одно денежное поле задано и не равно нулю, {@code false} в остальных случаях
     */
    public boolean hasAnyMoney() {
        return isNonZeroValue(sumUnits)
                || isNonZeroValue(sumRealMoney)
                || isNonZeroValue(chipsCost)
                || isNonZeroValue(chipsSpent)
                || isNonZeroValue(totalSum);
    }

    /**
     * Проверить что значение отлично от нуля.
     *
     * @param value значение для проверки
     * @return {@code false} если равно {@code null} или равно нулю (с любой точностью), {@code true} в остальных
     * случаях
     */
    private static boolean isNonZeroValue(BigDecimal value) {
        return value != null && value.compareTo(BigDecimal.ZERO) != 0;
    }
}
