package ru.yandex.direct.intapi.entity.balanceclient.repository;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.UpdateSetMoreStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.client.repository.ClientMapping;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsPaidByCertificate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.WalletCampaignsIsSumAggregated;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.entity.balanceclient.container.CampaignDataForNotifyOrder;
import ru.yandex.direct.intapi.entity.balanceclient.container.NotifyOrderDbCampaignChanges;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MULTICURRENCY_SUMS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CURRENCY_CONVERT_MONEY_CORRESPONDENCE;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.WALLET_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;

@Repository
public class NotifyOrderRepository {
    private static final Field<Boolean> IS_FIRST_AFTER_COPY_CONVERT_FIELD =
            DSL.field(CURRENCY_CONVERT_MONEY_CORRESPONDENCE.NEW_CID.isNotNull()
                    .and(CURRENCY_CONVERT_MONEY_CORRESPONDENCE.NEW_CID
                            .ne(CURRENCY_CONVERT_MONEY_CORRESPONDENCE.OLD_CID))
            ).as("first_after_copy_convert");
    private static final Field<String> EMAIL_FIELD =
            JooqMapperUtils.mysqlIf(
                    DSL.nvl(CAMP_OPTIONS.EMAIL, "").ne(""),
                    CAMP_OPTIONS.EMAIL,
                    USERS.EMAIL
            ).as("email");
    private static final Field<Boolean> IS_START_TIME_IN_FUTURE_FIELD =
            DSL.field(CAMPAIGNS.START_TIME.gt(DSL.currentLocalDate())).as("start_time_in_future");
    private static final Field<CampaignsCurrency> CURRENCY_FIELD =
            DSL.nvl(CAMPAIGNS.CURRENCY, CampaignsCurrency.YND_FIXED).as("currency");
    private static final Field<Long> START_TIME_TS_FIELD =
            SqlUtils.mysqlUnixTimestampForLocalDate(CAMPAIGNS.START_TIME).as("start_time_ts");
    private static final Field<ClientsWorkCurrency> CLIENTS_WORK_CURRENCY_FIELD =
            DSL.nvl(CLIENTS.WORK_CURRENCY, ClientsWorkCurrency.YND_FIXED).as("client_work_currency");
    private static final String WALLET_CAMPAIGNS_PARENT_ALIAS = "wallet_campaign_parent_alias";
    private static final Field<WalletCampaignsIsSumAggregated> SUM_AGGREGATED_FIELD =
            DSL.ifnull(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED,
                    WALLET_CAMPAIGNS.as(WALLET_CAMPAIGNS_PARENT_ALIAS).IS_SUM_AGGREGATED
            ).as("sum_aggregated_migration_alias");
    private final DslContextProvider dslContextProvider;

    @Autowired
    public NotifyOrderRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public CampaignDataForNotifyOrder fetchCampaignData(int shard, Long cid) {
        if (shard == NO_SHARD) {
            return null;
        }

        return dslContextProvider.ppc(shard)
                .select(
                        CAMPAIGNS.TYPE,
                        CAMPAIGNS.SUM,
                        CAMPAIGNS.SUM_BALANCE,
                        CAMPAIGNS.SUM_SPENT,
                        CAMPAIGNS.STATUS_MODERATE,
                        CAMPAIGNS.ORDER_ID,
                        CAMPAIGNS.NAME, CAMPAIGNS.UID,
                        CAMPAIGNS.STATUS_SHOW,
                        CAMPAIGNS.SUM_UNITS,
                        CAMPAIGNS.SUM_SPENT_UNITS,
                        CAMPAIGNS.CASHBACK,
                        CAMPAIGNS.TOTAL_CASHBACK,
                        CAMPAIGNS.BALANCE_TID,
                        CAMPAIGNS.STATUS_EMPTY,
                        CAMPAIGNS.ARCHIVED,
                        IS_START_TIME_IN_FUTURE_FIELD,
                        CAMPAIGNS.AGENCY_UID,
                        CAMPAIGNS.MANAGER_UID,
                        CAMP_OPTIONS.STATUS_POST_MODERATE,
                        CAMPAIGNS.PRODUCT_ID,
                        CAMPAIGNS.SOURCE,
                        CAMPAIGNS_MULTICURRENCY_SUMS.BALANCE_TID,
                        CAMPAIGNS_MULTICURRENCY_SUMS.SUM,
                        CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST,
                        CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_SPENT,
                        CURRENCY_FIELD,
                        IS_FIRST_AFTER_COPY_CONVERT_FIELD,
                        CAMPAIGNS.CID,
                        CAMPAIGNS.WALLET_CID,
                        USERS.CLIENT_ID,
                        CAMPAIGNS.AGENCY_ID,
                        CAMPAIGNS.CURRENCY_CONVERTED,
                        EMAIL_FIELD,
                        USERS.FIO,
                        USERS.LOGIN,
                        USERS.PHONE,
                        START_TIME_TS_FIELD,
                        CAMPAIGNS.FINISH_TIME,
                        USERS.LANG,
                        CAMPAIGNS.PAID_BY_CERTIFICATE,
                        SUM_AGGREGATED_FIELD,
                        WALLET_CAMPAIGNS.TOTAL_BALANCE_TID,
                        WALLET_CAMPAIGNS.TOTAL_SUM,
                        CLIENTS_WORK_CURRENCY_FIELD
                )
                .from(CAMPAIGNS)
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .leftJoin(USERS_OPTIONS).on(USERS_OPTIONS.UID.eq(USERS.UID))
                .leftJoin(CAMPAIGNS_MULTICURRENCY_SUMS).on(CAMPAIGNS_MULTICURRENCY_SUMS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(CURRENCY_CONVERT_MONEY_CORRESPONDENCE).on(
                        CAMPAIGNS.BALANCE_TID.eq(0L)
                                .and(CURRENCY_CONVERT_MONEY_CORRESPONDENCE.CLIENT_ID.eq(USERS.CLIENT_ID))
                                .and(CURRENCY_CONVERT_MONEY_CORRESPONDENCE.NEW_CID.eq(CAMPAIGNS.CID))
                )
                .leftJoin(WALLET_CAMPAIGNS.as(WALLET_CAMPAIGNS_PARENT_ALIAS)).on(
                        WALLET_CAMPAIGNS.as(WALLET_CAMPAIGNS_PARENT_ALIAS).WALLET_CID.eq(CAMPAIGNS.WALLET_CID)
                )
                .leftJoin(WALLET_CAMPAIGNS).on(
                        WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID)
                                .and(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                )
                .leftJoin(CLIENTS).on(CLIENTS.CLIENT_ID.eq(USERS.CLIENT_ID))
                .where(CAMPAIGNS.CID.eq(cid))
                .fetchOne(this::extractCampaignDataForNotifyOrder);
    }

    private CampaignDataForNotifyOrder extractCampaignDataForNotifyOrder(Record record) {
        CampaignsStatusshow campStatusShow = record.get(CAMPAIGNS.STATUS_SHOW);
        CampaignsStatusempty campStatusEmpty = record.get(CAMPAIGNS.STATUS_EMPTY);
        CampaignsArchived campArchived = record.get(CAMPAIGNS.ARCHIVED);
        CampaignsPaidByCertificate campPaidByCertificate = record.get(CAMPAIGNS.PAID_BY_CERTIFICATE);
        LocalDate campFinishTime = record.get(CAMPAIGNS.FINISH_TIME);
        return new CampaignDataForNotifyOrder()
                .withType(CampaignType.fromSource(record.get(CAMPAIGNS.TYPE)))
                .withSum(record.get(CAMPAIGNS.SUM))
                .withSumBalance(record.get(CAMPAIGNS.SUM_BALANCE))
                .withSumSpent(record.get(CAMPAIGNS.SUM_SPENT))
                .withStatusModerate(CampaignStatusModerate.fromSource(record.get(CAMPAIGNS.STATUS_MODERATE)))
                .withSource(CampaignSource.fromSource(record.get(CAMPAIGNS.SOURCE)))
                .withOrderId(record.get(CAMPAIGNS.ORDER_ID))
                .withName(record.get(CAMPAIGNS.NAME))
                .withUid(record.get(CAMPAIGNS.UID))
                .withStatusShow(campStatusShow == null ? null : campStatusShow == CampaignsStatusshow.Yes)
                .withSumUnits(record.get(CAMPAIGNS.SUM_UNITS))
                .withSumSpentUnits(record.get(CAMPAIGNS.SUM_SPENT_UNITS))
                .withCashback(record.get(CAMPAIGNS.CASHBACK))
                .withTotalCashback(record.get(CAMPAIGNS.TOTAL_CASHBACK))
                .withBalanceTid(record.get(CAMPAIGNS.BALANCE_TID))
                .withStatusEmpty(campStatusEmpty == null ? null : campStatusEmpty == CampaignsStatusempty.Yes)
                .withArchived(campArchived == null ? null : campArchived == CampaignsArchived.Yes)
                .withStartTimeInFuture(record.get(IS_START_TIME_IN_FUTURE_FIELD))
                .withAgencyUid(record.get(CAMPAIGNS.AGENCY_UID))
                .withManagerUid(record.get(CAMPAIGNS.MANAGER_UID))
                .withStatusPostModerate(
                        CampaignStatusPostmoderate.fromSource(record.get(CAMP_OPTIONS.STATUS_POST_MODERATE)))
                .withProductId(record.get(CAMPAIGNS.PRODUCT_ID))
                .withCmsBalanceTid(record.get(CAMPAIGNS_MULTICURRENCY_SUMS.BALANCE_TID))
                .withCmsSum(record.get(CAMPAIGNS_MULTICURRENCY_SUMS.SUM))
                .withCmsChipsCost(record.get(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST))
                .withCmsChipsSpent(record.get(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_SPENT))
                .withCurrency(CampaignMappings.currencyCodeFromDb(record.get(CURRENCY_FIELD)))
                .withFirstAfterCopyConvert(record.get(IS_FIRST_AFTER_COPY_CONVERT_FIELD))
                .withCampaignId(record.get(CAMPAIGNS.CID))
                .withWalletId(record.get(CAMPAIGNS.WALLET_CID))
                .withClientId(record.get(USERS.CLIENT_ID))
                .withAgencyId(record.get(CAMPAIGNS.AGENCY_ID))
                .withCurrencyConverted(
                        CampaignMappings.currencyConvertedFromDb(record.get(CAMPAIGNS.CURRENCY_CONVERTED)))
                .withEmail(record.get(CAMP_OPTIONS.EMAIL))
                .withFio(record.get(USERS.FIO))
                .withLogin(record.get(USERS.LOGIN))
                .withPhone(record.get(USERS.PHONE))
                .withStartTimeTs(record.get(START_TIME_TS_FIELD))
                .withFinishDate(campFinishTime)
                .withLang(record.get(USERS.LANG))
                .withPaidByCertificate(
                        campPaidByCertificate == null ? null : campPaidByCertificate == CampaignsPaidByCertificate.Yes)
                .withWalletAggregateMigrated(AggregatingSumStatus.fromSource(record.get(SUM_AGGREGATED_FIELD)))
                .withTotalBalanceTid(record.get(WALLET_CAMPAIGNS.TOTAL_BALANCE_TID))
                .withTotalSum(record.get(WALLET_CAMPAIGNS.TOTAL_SUM))
                .withClientWorkCurrency(ClientMapping.workCurrencyFromDb(record.get(CLIENTS_WORK_CURRENCY_FIELD)));
    }

    public List<BsResyncItem> fetchCampaignItemsForBsResync(int shard, Long campaignId, BsResyncPriority priority) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.CID, PHRASES.PID, BANNERS.BID)
                .from(PHRASES)
                .join(BANNERS).on(BANNERS.PID.eq(PHRASES.PID))
                .where(PHRASES.CID.eq(campaignId)
                        .and(PHRASES.STATUS_BS_SYNCED.eq(PhrasesStatusbssynced.Yes)
                                .or(BANNERS.STATUS_BS_SYNCED.eq(BannersStatusbssynced.Yes)))
                        .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                        .and(BANNERS.BANNER_ID.gt(0L)))
                .fetch(record -> new BsResyncItem(priority, record.get(PHRASES.CID), record.get(BANNERS.BID),
                        record.get(PHRASES.PID)));
    }

    /**
     * Обновить запись в таблице campaigns по cid, balance_tid и wallet_cid
     *
     * @param balanceTid {@code campaigns.balance_tid} прочитанный из БД в начале обработки нотификации
     * @param walletId   {@code campaigns.wallet_cid} прочитанный из БД в начале обработки нотификации
     * @return true, если запись была успешно обновлена, иначе false
     */
    public boolean updateCampaignData(DSLContext dslContext, Long campaignId, Long balanceTid, Long walletId,
                                      NotifyOrderDbCampaignChanges changes) {
        UpdateSetMoreStep<CampaignsRecord> updateStep = dslContext
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now());

        if (changes.isPaidByCertificate()) {
            updateStep.set(CAMPAIGNS.PAID_BY_CERTIFICATE, CampaignsPaidByCertificate.Yes);
        }

        if (changes.needResetStatusBsSynced()) {
            updateStep.set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No);
        }

        if (changes.getBalanceTid() != null) {
            updateStep.set(CAMPAIGNS.BALANCE_TID, changes.getBalanceTid());
        }

        if (changes.getStatusMail() != null) {
            updateStep.set(CAMPAIGNS.STATUS_MAIL, changes.getStatusMail().getNumeric());
        }

        if (changes.getSum() != null) {
            updateStep.set(CAMPAIGNS.SUM, changes.getSum());
        }

        if (changes.getSumBalance() != null) {
            updateStep.set(CAMPAIGNS.SUM_BALANCE, changes.getSumBalance());
        }

        if (changes.getSumSpent() != null) {
            updateStep.set(CAMPAIGNS.SUM_SPENT, changes.getSumSpent());
        }

        if (changes.getSumUnits() != null) {
            updateStep.set(CAMPAIGNS.SUM_UNITS, changes.getSumUnits());
        }

        if (changes.getSumLast() != null) {
            updateStep.set(CAMPAIGNS.SUM_LAST, changes.getSumLast());
        }

        if (changes.getSumToPay() != null) {
            updateStep.set(CAMPAIGNS.SUM_TO_PAY, changes.getSumToPay());
        }

        if (changes.getCashback() != null) {
            updateStep.set(CAMPAIGNS.CASHBACK, changes.getCashback());
        }

        if (changes.getTotalCashback() != null) {
            updateStep.set(CAMPAIGNS.TOTAL_CASHBACK, changes.getTotalCashback());
        }

        return updateStep
                .where(CAMPAIGNS.CID.eq(campaignId)
                        .and(CAMPAIGNS.WALLET_CID.eq(walletId))
                        .and(CAMPAIGNS.BALANCE_TID.eq(balanceTid)))
                .execute() > 0;
    }

    /**
     * Вернуть истину, если под кошельком существует хотя бы одна кампания, запускающаяся в будущем
     *
     * @param shard    шард
     * @param walletId номер кошелька
     * @param userId   идентификатор пользователя
     */
    public boolean isThereAnyCampStartingInFutureUnderWallet(int shard, long walletId, long userId) {
        return dslContextProvider.ppc(shard)
                .select(DSL.val(1))
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.WALLET_CID.eq(walletId)
                        .and(CAMPAIGNS.UID.eq(userId))
                        .and(CAMPAIGNS.START_TIME.gt(LocalDate.now())))
                .limit(1)
                .fetch()
                .isNotEmpty();
    }
}
