package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.agency.model.AgencyAdditionalCurrency;
import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyAgencyAdditionalCurrenciesItem;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyAgencyAdditionalCurrenciesParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@ParametersAreNonnullByDefault
public class NotifyAgencyAdditionalCurrenciesService {
    static final int INVALID_INPUT_ERROR_CODE = 2017;
    static final int CURRENCY_DISAPPEARED_ERROR_CODE = 2018;
    private static final Logger logger = LoggerFactory.getLogger(NotifyAgencyAdditionalCurrenciesService.class);

    private final AgencyService agencyService;
    private final ClientService clientService;

    @Autowired
    public NotifyAgencyAdditionalCurrenciesService(AgencyService agencyService, ClientService clientService) {
        this.agencyService = agencyService;
        this.clientService = clientService;
    }

    /**
     * Обработать агентство, описываемое переданными параметрами
     */
    public BalanceClientResponse processAgency(NotifyAgencyAdditionalCurrenciesParameters parameters) {
        if (!inputParamsAreValid(parameters)) {
            return BalanceClientResponse.error(INVALID_INPUT_ERROR_CODE, "Invalid input");
        }
        Long clientId = parameters.getClientId();

        if (clientService.getClient(ClientId.fromLong(clientId)) == null) {
            // Если мы не знаем клиента - просто завершаем обработку
            logger.warn("Got notification for unknown client with id {}", clientId);
            return BalanceClientResponse.success();
        }

        logger.info("Processing {} additional currencies for agency with client id {}",
                parameters.getAdditionalCurrencies().size(), clientId);
        List<AgencyAdditionalCurrency> newAdditionalCurrencies = mapList(parameters.getAdditionalCurrencies(),
                i -> itemToModel(clientId, i));

        Map<CurrencyCode, LocalDate> oldCurrencyToDate =
                getCurrencyCodeToExpireDate(agencyService.getAllAdditionalCurrencies(clientId));

        // удалений записей получаться не должно (в худшем случае у записи изменится на меньшее время,
        // до которого она действительна), но на всякий случай всё-равно будем проверять, что записи никуда не исчезли
        Map<CurrencyCode, LocalDate> newCurrencyToDate = getCurrencyCodeToExpireDate(newAdditionalCurrencies);
        if (!newCurrencyToDate.keySet().containsAll(oldCurrencyToDate.keySet())) {
            Set<CurrencyCode> oldCurrencies = oldCurrencyToDate.keySet();
            oldCurrencies.removeAll(newCurrencyToDate.keySet());
            String msg =
                    String.format("Entries for currencies %s disappeared for agency with client id %s", oldCurrencies,
                            clientId);
            logger.error(msg);
            return BalanceClientResponse.error(CURRENCY_DISAPPEARED_ERROR_CODE, msg);
        }

        List<AgencyAdditionalCurrency> additionalCurrenciesToUpdate =
                filterList(newAdditionalCurrencies, ac -> !currencyNotChanged(ac, oldCurrencyToDate));

        if (!additionalCurrenciesToUpdate.isEmpty()) {
            logger.info("Adding/updating data {} for agency with client id {}",
                    additionalCurrenciesToUpdate, clientId);
            int num = agencyService.addAdditionalCurrencies(additionalCurrenciesToUpdate);
            logger.info("Successfully updated {} entries", num);
        } else {
            logger.info("Nothing to change for agency with client id {}", clientId);
        }
        return BalanceClientResponse.success();
    }

    boolean currencyNotChanged(AgencyAdditionalCurrency item, Map<CurrencyCode, LocalDate> oldCurrencyToDate) {
        return oldCurrencyToDate.containsKey(item.getCurrencyCode()) &&
                oldCurrencyToDate.get(item.getCurrencyCode()).equals(item.getExpirationDate());
    }

    boolean inputParamsAreValid(@Nullable NotifyAgencyAdditionalCurrenciesParameters parameters) {
        if (parameters == null) {
            return false;
        }
        ItemValidationBuilder<NotifyAgencyAdditionalCurrenciesParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters, Defect.class);

        validationBuilder.item(parameters.getClientId(), "clientId")
                .check(notNull())
                .check(validId(), When.notNull());

        validationBuilder.list(parameters.getAdditionalCurrencies(), "currencies")
                .check(notNull())
                .check(eachNotNull(), When.notNull())
                .checkEach(additionalCurrencyItemIsCorrect(), When.notNull());

        ValidationResult<NotifyAgencyAdditionalCurrenciesParameters, Defect> result =
                validationBuilder.getResult();

        if (result.hasAnyErrors()) {
            logger.error("Input validation failed: {}", result.flattenErrors());
            return false;
        }
        return true;
    }

    private static Constraint<NotifyAgencyAdditionalCurrenciesItem, Defect> additionalCurrencyItemIsCorrect() {
        Predicate<NotifyAgencyAdditionalCurrenciesItem> predicate =
                item -> item.getExpireDate() != null && item.getCurrencyCode() != null && CurrencyCode
                        .isRealCurrencyCode(item.getCurrencyCode());
        return Constraint.fromPredicate(predicate, CommonDefects.invalidValue());
    }

    private Map<CurrencyCode, LocalDate> getCurrencyCodeToExpireDate(
            List<AgencyAdditionalCurrency> additionalCurrencies) {
        return additionalCurrencies
                .stream()
                .collect(Collectors.toMap(AgencyAdditionalCurrency::getCurrencyCode,
                        AgencyAdditionalCurrency::getExpirationDate));
    }

    private AgencyAdditionalCurrency itemToModel(Long clientId,
                                                 NotifyAgencyAdditionalCurrenciesItem additionalCurrency) {
        return new AgencyAdditionalCurrency()
                .withClientId(clientId)
                .withCurrencyCode(CurrencyCode.valueOf(additionalCurrency.getCurrencyCode()))
                .withExpirationDate(additionalCurrency.getExpireDate());
    }
}
