package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters;
import ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyClientCashBackValidationService;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис обработки нотификаций о кэшбеке клиента.
 *
 * @see <a href="https://st.yandex-team.ru/BALANCE-34923">BALANCE-34923: [Директ] Cashback</a>
 */
@ParametersAreNonnullByDefault
@Service
public class NotifyClientCashBackService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyClientCashBackService.class);

    private final ClientService clientService;
    private final NotifyClientCashBackValidationService validationService;
    private final CashbackNotificationsService cashbackNotificationsService;

    static final String WRONG_CURRENCY_ERROR_MESSAGE =
            "Got wrong currency %s from Balance for client with id %d and currency %s";
    static final int WRONG_CURRENCY_ERROR_CODE = 4001;

    @Autowired
    public NotifyClientCashBackService(ClientService clientService,
                                       NotifyClientCashBackValidationService validationService,
                                       CashbackNotificationsService cashbackNotificationsService) {
        this.clientService = clientService;
        this.validationService = validationService;
        this.cashbackNotificationsService = cashbackNotificationsService;
    }

    /**
     * Обработчик нотификации.
     *
     * @param updateRequest балансовая нотификация
     * @return ответ балансу: успех или ошибка
     */
    public BalanceClientResponse notifyClientCashBack(NotifyClientCashBackParameters updateRequest) {
        logger.info(updateRequest.toString());
        // Базовая валидация входных данных
        BalanceClientResponse validationResponse;
        validationResponse = validationService.validateRequest(updateRequest);
        if (validationResponse != null) {
            return validationResponse;
        }

        // Дополнительные проверки
        ClientId clientId = ClientId.fromLong(updateRequest.getClientId());
        var client = clientService.getClient(clientId);
        if (client == null) {
            // Если мы не знаем клиента - завершаем обработку
            logger.warn("Got notification for unknown client with id {}", clientId);
            return BalanceClientResponse.success();
        }
        CurrencyCode currencyCode = Currencies.getCurrency(updateRequest.getBalanceCurrency()).getCode();
        if (!client.getWorkCurrency().equals(currencyCode)) {
            // Если валюта в запросе не совпадает с валютой клиента - возвращаем ошибку
            String message = String.format(WRONG_CURRENCY_ERROR_MESSAGE, currencyCode,
                    clientId.asLong(),
                    clientService.getWorkCurrency(clientId).getBalanceCurrencyName());
            logger.warn(message);
            return BalanceClientResponse.error(WRONG_CURRENCY_ERROR_CODE, message);
        }

        BigDecimal bonus = checkBonus(updateRequest.getCashbackConsumedBonus(), clientId);
        BigDecimal awaitingBonus = checkBonus(updateRequest.getCashBackBonus(), clientId);
        clientService.updateClientCashBackBonus(clientId, bonus, awaitingBonus);

        // Считаем начисленный бонус — у клиента сохраняется общий начисленный за всё время
        // поэтому вычитаем предыдущее значение из нового
        BigDecimal prevBonus = nvl(client.getCashBackBonus(), BigDecimal.ZERO);
        BigDecimal addedBonus = checkBonus(bonus.subtract(prevBonus), clientId);

        // Уведомление отправляем, только если сумма начисления больше рубля, чтобы обрезать микроначисления
        if (addedBonus.compareTo(BigDecimal.ONE) > 0) {
            cashbackNotificationsService.addSmsNotification(clientId, addedBonus, currencyCode);
        }
        return BalanceClientResponse.success();
    }

    private static BigDecimal checkBonus(BigDecimal bonus, ClientId clientId) {
        // в редких случаях из-за ошибок округления Баланс может прислать отрицательный кэшбек
        // заменяем  нулевым и пишем предупреждение в лог
        // Уведомления клиенту при этом не отправляются
        if (bonus.compareTo(BigDecimal.ZERO) < 0) {
            logger.warn("Got notification with negative cash back bonus {} for client with id {}", bonus.longValue(),
                    clientId);
            return BigDecimal.ZERO;
        }
        return bonus;
    }
}
