package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.autooverdraftmail.repository.ClientOverdraftLimitChangesRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientParameters;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.ValidationUtils;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singleton;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class NotifyClientService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyClientService.class);
    private static final int INVALID_OVERDRAFT_LIMIT_ERROR_CODE = 1010;
    private static final int INVALID_OVERDRAFT_SPENT_ERROR_CODE = 1011;
    private static final int CLIENT_CURRENCY_CONVERSION_IN_PROGRESS_ERROR_CODE = 1012;

    private final ClientService clientService;
    private final UserService userService;
    private final RbacService rbacService;
    private final ShardSupport shardSupport;
    private final CampaignRepository campaignRepository;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final ClientOverdraftLimitChangesRepository overdraftLimitChangesRepository;

    @Autowired
    public NotifyClientService(
            ClientService clientService, BsResyncQueueRepository bsResyncQueueRepository,
            CampaignRepository campaignRepository, ShardSupport shardSupport, RbacService rbacService,
            UserService userService,
            ClientOverdraftLimitChangesRepository overdraftLimitChangesRepository
    ) {
        this.clientService = clientService;
        this.userService = userService;
        this.rbacService = rbacService;
        this.shardSupport = shardSupport;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.campaignRepository = campaignRepository;
        this.overdraftLimitChangesRepository = overdraftLimitChangesRepository;
    }

    public BalanceClientResponse notifyClient(NotifyClientParameters updateRequest) {
        ClientId clientId = ClientId.fromNullableLong(updateRequest.getClientId());
        BalanceClientResponse clientIdValidationResult = validateClientId(updateRequest);

        if (clientIdValidationResult != null) {
            return clientIdValidationResult;
        }

        if (clientId == null || !userService.clientExists(clientId.asLong())) {
            return BalanceClientResponse.success(String.format("ClientID %s is not known", clientId));
        }

        if (Boolean.TRUE.equals(updateRequest.getMigrateToCurrencyDone())) {
            clientService.setBalanceSideConvertFinished(clientId);
        }

        if (clientService.isClientConvertingSoonAndNotWaitingOverdraftNow(clientId)) {
            return BalanceClientResponse.error(CLIENT_CURRENCY_CONVERSION_IN_PROGRESS_ERROR_CODE, String.format(
                    "Client %s is going to currency convert soon, will accept notifications after it's done",
                    clientId));
        }

        CurrencyCode workCurrency = clientService.getWorkCurrency(clientId).getCode();
        BalanceClientResponse currencyValidationResult = validateWorkCurrency(updateRequest, workCurrency);
        if (currencyValidationResult != null) {
            return currencyValidationResult;
        }

        BalanceClientResponse moneyValidationResult = validateMoneyAmountFields(updateRequest);
        if (moneyValidationResult != null) {
            return moneyValidationResult;
        }

        if (updateRequest.isBusinessUnit() != null) {
            // записи в clients может не быть, поэтому получаем "умолчание" через Optional'ы
            Boolean clientIsBusinessUnit = Optional.ofNullable(clientService.isBusinessUnitClient(clientId))
                    .orElse(false);
            if (updateRequest.isBusinessUnit() != clientIsBusinessUnit) {
                //если изменился флаг is_business_unit, добавляем все кампании клиента на переотправку
                resyncClientCampaignsWithBs(clientId);
            }
        }

        Boolean balanceBanned =
                updateRequest.getStatusBalanceBanned() == null ? Boolean.FALSE : updateRequest.getStatusBalanceBanned();

        Client client = clientService.getClient(clientId);
        boolean isAutoOverdraftSet = client != null && client.getAutoOverdraftLimit() != null
                && client.getAutoOverdraftLimit().compareTo(BigDecimal.ZERO) > 0;
        if (isAutoOverdraftSet) {
            boolean clientExplicitlyBannedFlagChanged = !balanceBanned.equals(clientService.isBannedClient(clientId));
            boolean overdraftLimitWasOrWillBeZero = client.getOverdraftLimit().compareTo(BigDecimal.ZERO) == 0
                    || updateRequest.getOverdraftLimit().compareTo(BigDecimal.ZERO) == 0;
            boolean overdraftLimitChanged =
                    client.getOverdraftLimit().compareTo(updateRequest.getOverdraftLimit()) != 0;
            boolean clientImplicitlyBannedFlagChanged = overdraftLimitChanged && overdraftLimitWasOrWillBeZero;
            boolean debtChanged = updateRequest.getOverdraftSpent()
                    .setScale(Money.MONEY_CENT_SCALE, RoundingMode.HALF_UP)
                    .subtract(client.getDebt())
                    .abs()
                    .compareTo(BigDecimal.ZERO) > 0;
            if (clientExplicitlyBannedFlagChanged || clientImplicitlyBannedFlagChanged || debtChanged) {
                resyncPersonalWalletsWithBs(clientId);
            }
        } else if (client != null // клиенту повторно стал доступен лимит овердрафта, запомним клиента для рассылки
                && !Boolean.TRUE.equals(clientService.isBannedClient(clientId))
                && client.getOverdraftLimit().compareTo(BigDecimal.ZERO) == 0
                && updateRequest.getOverdraftLimit().compareTo(BigDecimal.ZERO) > 0
                && client.getAutoOverdraftNotified().equals(Boolean.TRUE)) {
            overdraftLimitChangesRepository.add(shardSupport.getShard(ShardKey.CLIENT_ID, clientId), clientId, false);
        }

        clientService.updateClientOptions(clientId, updateRequest.getTid(),
                updateRequest.getOverdraftLimit(), updateRequest.getOverdraftSpent(),
                updateRequest.getNextPayDate(), balanceBanned,
                updateRequest.isNonResident(), updateRequest.isBusinessUnit(), updateRequest.isBrand());

        if (Boolean.TRUE.equals(updateRequest.getCanBeForceCurrencyConverted())) {
            updateForceCurrencyConversionStatus(clientId, workCurrency);
        }
        return BalanceClientResponse.success();
    }

    private static BalanceClientResponse validateClientId(NotifyClientParameters parameters) {
        ItemValidationBuilder<NotifyClientParameters, IntApiDefect> v =
                ItemValidationBuilder.of(parameters, IntApiDefect.class);
        v.item(parameters.getClientId(), NotifyClientParameters.CLIENT_ID_FIELD_NAME)
                .check(notNull())
                .check(validId(), When.notNull());

        ValidationResult<NotifyClientParameters, IntApiDefect> vr = v.getResult();
        if (vr.hasAnyErrors()) {
            return BalanceClientResponse.criticalError(ValidationUtils.getErrorText(vr));
        }
        return null;
    }

    private void resyncPersonalWalletsWithBs(ClientId clientId) {
        long clientChiefUid = rbacService.getChiefByClientId(clientId);
        int shard = shardSupport.getShard(ShardKey.UID, clientChiefUid);
        List<CampaignSimple> walletsSimple =
                campaignRepository.searchNotEmptyNotArchivedCampaigns(shard, clientId, singleton(CampaignType.WALLET));
        List<Campaign> walletCampaigns = campaignRepository.getClientsCampaignsByIds(shard, clientId,
                mapList(walletsSimple, CampaignSimple::getId));
        Collection<Long> cids2Resync = mapList(
                filterList(walletCampaigns, c -> c.getAgencyId() == null || c.getAgencyId() == 0), Campaign::getId);
        campaignRepository.resetBannerSystemSyncStatus(shard, cids2Resync);
    }

    private void resyncClientCampaignsWithBs(ClientId clientId) {
        int shard = shardSupport.getShard(ShardKey.CLIENT_ID, clientId);
        Collection<Long> cids2Resync = campaignRepository.getNonArchivedCampaignIdsWhichWereInBs(shard, clientId);
        logger.info("Add campaigns in bs_resync on is_business_unit change: {}", cids2Resync);
        Collection<BsResyncItem> addCampaigns2Resync = mapList(cids2Resync,
                cid -> new BsResyncItem(BsResyncPriority.DEFAULT, cid));
        bsResyncQueueRepository.addToResync(shard, addCampaigns2Resync);
    }

    /**
     * Отмечает клиента как принудительного конвертируемого, если это действительно необходимо
     *
     * @param clientId
     * @param workCurrency
     */
    private void updateForceCurrencyConversionStatus(ClientId clientId, CurrencyCode workCurrency) {
        boolean needSave = false;
        if (CurrencyCode.YND_FIXED.equals(workCurrency)) {
            needSave = true;
        } else {
            long chiefUid = rbacService.getChiefByClientId(clientId);
            RbacRole uidRole = rbacService.getUidRole(chiefUid);
            if (uidRole.equals(RbacRole.AGENCY)) {
                needSave = true;
            }
        }
        if (needSave) {
            clientService.forceCurrencyConversion(clientId);
        }
    }

    private static BalanceClientResponse validateWorkCurrency(NotifyClientParameters updateRequest,
                                                              CurrencyCode workCurrency) {
        if (isPositive(updateRequest.getOverdraftLimit()) || isPositive(updateRequest.getOverdraftSpent())
                || updateRequest.getNextPayDate() != null || updateRequest.getBalanceCurrency() != null) {
            String resolvedRequestCurrencyName = updateRequest.getBalanceCurrency() == null
                    ? CurrencyCode.YND_FIXED.name()
                    : updateRequest.getBalanceCurrency();
            if (!workCurrency.name().equals(resolvedRequestCurrencyName)) {
                return BalanceClientResponse.criticalError(
                        String.format("Currency from balance %s doesn't match client currency %s for ClientID",
                                resolvedRequestCurrencyName, workCurrency.name()));
            }
        }
        return null;
    }

    private static BalanceClientResponse validateMoneyAmountFields(NotifyClientParameters parameters) {
        BalanceClientResponse result = validateMoneyAmountField("Overdraft limit",
                parameters.getOverdraftLimit(), INVALID_OVERDRAFT_LIMIT_ERROR_CODE);
        if (result != null) {
            return result;
        }
        return validateMoneyAmountField("Debt", parameters.getOverdraftSpent(), INVALID_OVERDRAFT_SPENT_ERROR_CODE);
    }

    @Nullable
    private static BalanceClientResponse validateMoneyAmountField(String fieldName, BigDecimal amount, int code) {
        if (amount == null || amount.signum() < 0) {
            String formattedValue = amount == null ? "undef" : amount.stripTrailingZeros().toPlainString();
            return BalanceClientResponse.error(code,
                    String.format("%s from balance: %s, must be greater than zero", fieldName, formattedValue));
        }
        return null;
    }

    private static boolean isPositive(BigDecimal value) {
        return value != null && value.signum() > 0;
    }
}
