package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyOrder;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampActivizationRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.WhenMoneyOnCampWasEvents;
import ru.yandex.direct.core.entity.client.repository.AgencyClientRelationRepository;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.balanceclient.container.CampaignDataForNotifyOrder;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.intapi.entity.balanceclient.repository.NotifyOrderRepository;
import ru.yandex.direct.intapi.entity.balanceclient.service.migration.MigrationSchema;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.db.PpcPropertyNames.AUTOBUDGET_RESTART_SEND_TO_BS_ON_NEW_MONEY;
import static ru.yandex.direct.common.db.PpcPropertyNames.CHECK_CASHBACK_ONLY_ON_NEW_MONEY;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
@Service
public class NotifyOrderCampaignPostProcessingService {
    // через 90 дней БК архивирует ресурсы
    static final Duration BS_RESOURCE_ARC_PERIOD = Duration.ofDays(90);
    private static final Logger logger = LoggerFactory.getLogger(NotifyOrderCampaignPostProcessingService.class);
    private final CampaignService campaignService;
    private final NotifyOrderNotificationService notifyOrderNotificationService;

    private final NotifyOrderRepository notifyOrderRepository;
    private final CampaignRepository campaignRepository;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final AgencyClientRelationRepository agencyClientRelationRepository;
    private final UserRepository userRepository;
    private final CampActivizationRepository campActivizationRepository;
    private final OrderStatService orderStatService;
    private final PpcProperty<Boolean> sendToBsOnRefillProp;
    private final PpcProperty<Boolean> checkCashbackOnlyProp;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public NotifyOrderCampaignPostProcessingService(NotifyOrderRepository notifyOrderRepository,
                                                    CampaignRepository campaignRepository,
                                                    CampaignService campaignService,
                                                    NotifyOrderNotificationService notifyOrderNotificationService,
                                                    BsResyncQueueRepository bsResyncQueueRepository,
                                                    AgencyClientRelationRepository agencyClientRelationRepository,
                                                    UserRepository userRepository,
                                                    CampActivizationRepository campActivizationRepository,
                                                    OrderStatService orderStatService,
                                                    PpcPropertiesSupport ppcPropertiesSupport) {
        this.notifyOrderRepository = notifyOrderRepository;
        this.campaignRepository = campaignRepository;
        this.campaignService = campaignService;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.agencyClientRelationRepository = agencyClientRelationRepository;
        this.userRepository = userRepository;
        this.campActivizationRepository = campActivizationRepository;
        this.notifyOrderNotificationService = notifyOrderNotificationService;
        this.orderStatService = orderStatService;
        this.sendToBsOnRefillProp = ppcPropertiesSupport.get(
                AUTOBUDGET_RESTART_SEND_TO_BS_ON_NEW_MONEY,
                Duration.ofMinutes(5)
        );
        this.checkCashbackOnlyProp = ppcPropertiesSupport.get(
                CHECK_CASHBACK_ONLY_ON_NEW_MONEY,
                Duration.ofMinutes(5)
        );
    }

    /**
     * Обработать кампанию, у которой изменилась только стоимость/количество фишек
     *
     * @param shard      шард
     * @param campaignId идентификатор кампании
     */
    void processUnchangedCampaign(int shard, long campaignId) {
        // Лениво переотправляем в БК
        List<BsResyncItem> resyncData =
                singletonList(new BsResyncItem(BsResyncPriority.MULTICURRENCY_SUMS_UPDATED, campaignId));
        bsResyncQueueRepository.addToResync(shard, resyncData);
    }

    /**
     * Обработать кампанию, на счету которой не было денег, в случае, если в текущей нотификации они поступили
     *
     * @param shard          шард
     * @param dbCampaignData данные кампании, полученные из базы данных
     * @param campsInWallet  список кампаний под общим счетом; заполняется, если кампания с заданным идентификатором
     *                       является общим счетом
     */
    void processMoneyRefillFromZero(int shard, CampaignDataForNotifyOrder dbCampaignData,
                                    List<CampaignForNotifyOrder> campsInWallet) {
        if (dbCampaignData.getType() == CampaignType.WALLET && !campsInWallet.isEmpty()) {
            // переотправляем в БК кампании под общим счетом без OrderID (не черновики), если на счете появились деньги
            // это нужно, по двум причинам:
            // - чтобы заказ, созданный во время отсутствия денег на ОС создался в БК
            // - чтобы перезапустился автобюджет (если надо)
            resyncCampaignsUnderWalletInBannerSystem(shard, dbCampaignData, campsInWallet);
        }

        /*
        В перле в этом месте был код, который апдейтил поле LastChange баннерам кампании или кампании под общим счетом,
        для кампаний с statusYacobotDeleted. Т.к. этот статус выпилили, то и этот код перестал иметь смысл.
         */

        // если показы по кампании закончились более X дней назад, БК может заархивировать ресурсы,
        // поэтому мы должны перепослать баннеры и условия
        unarchiveOldCampaignResources(shard, dbCampaignData, campsInWallet);
    }

    /**
     * Перепослать в БК старые баннеры и условия показа. Проверка возраста производится по статистике, полученной из БК,
     * так как campaigns.lastShowTime к сожалению не подходит. Архивные и выключенные кампании игнорируем и не
     * переотправляем
     *
     * @param shard          шард
     * @param dbCampaignData данные кампании, полученные из базы данных
     * @param campsInWallet  список кампаний под общим счетом; заполняется, если кампания с заданным идентификатором
     *                       является общим счетом
     */
    void unarchiveOldCampaignResources(int shard, CampaignDataForNotifyOrder dbCampaignData,
                                       List<CampaignForNotifyOrder> campsInWallet) {
        Map<Long, Long> ordersForBs;
        if (dbCampaignData.getType() == CampaignType.WALLET) {
            ordersForBs = campsInWallet.stream()
                    .filter(c -> isValidId(c.getOrderId()))
                    .filter(CampaignForNotifyOrder::getStatusShow)
                    .filter(c -> !c.getStatusArchived())
                    .collect(Collectors.toMap(CampaignForNotifyOrder::getId, CampaignForNotifyOrder::getOrderId));
        } else {
            ordersForBs = new HashMap<>();
            if (isValidId(dbCampaignData.getOrderId())) {
                ordersForBs.put(dbCampaignData.getCampaignId(), dbCampaignData.getOrderId());
            }
        }

        List<Long> ordersIds = new ArrayList<>(ordersForBs.values());
        Map<Long, LocalDate> ordersToLastDay = orderStatService.getLastDayOfCampaigns(ordersIds);
        for (Map.Entry<Long, Long> entry : ordersForBs.entrySet()) {
            Long localCampaignId = entry.getKey();
            Long orderId = entry.getValue();
            LocalDate lastShowDate = ordersToLastDay.get(orderId);
            if (lastShowDate == null
                    || lastShowDate.compareTo(LocalDate.now().minusDays(BS_RESOURCE_ARC_PERIOD.toDays())) <= 0) {
                List<BsResyncItem> resyncData =
                        notifyOrderRepository.fetchCampaignItemsForBsResync(shard, localCampaignId,
                                BsResyncPriority.UNARC_CAMP_IN_BS_ON_NOTIFY_ORDER2);
                bsResyncQueueRepository.addToResync(shard, resyncData);
                logger.info("Added {} banners into bs_resync_queue for campaign {} / OrderID: {}",
                        resyncData.size(), localCampaignId, orderId);
            }
        }
    }

    /**
     * Переотправляем в БК кампании под кошельком, которые были созданы, когда на кошельке не было денег и не отправлены
     * или были отправлены, но со statusActive=No или для того, чтобы перезапусть автобюджет
     * <p>
     * Кампания, переданная в этот метод обязательно должна быть кошельком.
     *
     * @param shard          шард
     * @param dbCampaignData данные кампании, полученные из базы данных
     * @param campsInWallet  список кампаний под общим счетом; заполняется, если кампания с заданным идентификатором
     *                       является общим счетом
     */
    void resyncCampaignsUnderWalletInBannerSystem(int shard, CampaignDataForNotifyOrder dbCampaignData,
                                                  List<CampaignForNotifyOrder> campsInWallet) {
        var sendToBsOnRefill = sendToBsOnRefillProp.getOrDefault(false);

        List<Long> newCidsForBannerSystem = campsInWallet.stream()
                .filter(c -> sendToBsOnRefill
                        || !isValidId(c.getOrderId()) && !CampaignStatusModerate.NEW.equals(c.getStatusModerate())
                )
                .map(CampaignForNotifyOrder::getId)
                .collect(Collectors.toList());
        if (!newCidsForBannerSystem.isEmpty()) {
            logger.info("Resetting statusBsSynced for campaigns connected to wallet {}: {}",
                    dbCampaignData.getCampaignId(), newCidsForBannerSystem);
            // исходный кошелёк находится в том же шарде, что и кампании под ним
            campaignRepository.resetBannerSystemSyncStatus(shard, newCidsForBannerSystem);
        }

        // ставим в очередь на ожидание активизации заказы, которые уже были в БК
        // есть шанс, что мы отправляли их когда денег не было и записали себе statusActive=No
        // с появлением денег показы могут возобновиться, надо дождаться активизации и обновить statusActive
        Set<Long> cidsForActivization = campsInWallet.stream()
                .filter(c -> isValidId(c.getOrderId()))
                .filter(CampaignForNotifyOrder::getStatusShow)
                .map(CampaignForNotifyOrder::getId)
                .collect(Collectors.toSet());
        if (!cidsForActivization.isEmpty()) {
            logger.info("Adding campaigns connected to wallet {} to camp_activization queue: {}",
                    dbCampaignData.getCampaignId(), cidsForActivization);
            campActivizationRepository.addCampsForActivization(shard, cidsForActivization);
        }
    }

    /**
     * Обработать кампанию, на которой изменилась зачисленная сумма денег
     *
     * @param shard          шард
     * @param updateRequest  полученный запрос на изменение кампании
     * @param dbCampaignData данные кампании, полученные из базы данных
     * @param productRate    кол-во оплачиваемых единиц продукта, к которому относится кампания
     * @param sum            новая сумма на кампании
     * @param sumDelta       разница между старой и новой суммой
     */
    void processSumOnCampChange(
            int shard, NotifyOrderParameters updateRequest,
            CampaignDataForNotifyOrder dbCampaignData, Long productRate, Money sum, Money sumDelta,
            MigrationSchema.State state
    ) {
        campaignRepository.setCampOptionsLastPayTimeNow(shard, dbCampaignData.getCampaignId());

        // В перле округление делается так: my $sum_payed = sprintf("%.2f", $sum_delta);
        Money sumPayed = Money.valueOf(
                sumDelta.bigDecimalValue().setScale(2, RoundingMode.HALF_DOWN),
                sumDelta.getCurrencyCode()
        );
        if (sumPayed.greaterThanZero() && !isValidId(dbCampaignData.getWalletId())) {
            campaignService.whenMoneyOnCampWas(dbCampaignData.getCampaignId(), WhenMoneyOnCampWasEvents.MONEY_IN);
        }

        if (dbCampaignData.getType() == CampaignType.WALLET) {
            // для общего счета считаем, что под ним есть хотя бы одна кампания, которая запустится в будущем
            dbCampaignData.withStartTimeInFuture(
                    notifyOrderRepository.isThereAnyCampStartingInFutureUnderWallet(shard,
                            dbCampaignData.getCampaignId(), dbCampaignData.getUid()));
        }

        // Разархивировать клиента
        if (!sumPayed.isZero()) {
            if (isValidId(dbCampaignData.getAgencyId())) {
                agencyClientRelationRepository
                        .unarchiveClients(shard, ClientId.fromLong(dbCampaignData.getAgencyId()),
                                singleton(ClientId.fromLong(dbCampaignData.getClientId())));
                userRepository
                        .updateLastChange(shard, singleton(dbCampaignData.getUid()), LocalDateTime.now());
            } else if (isValidId(dbCampaignData.getManagerUid())) {
                userRepository.unarchiveUsers(shard, singleton(dbCampaignData.getUid()));
            }
        }

        boolean isCashbackOnly = checkCashbackOnlyProp.getOrDefault(false) &&
                isCashbackOnly(updateRequest, dbCampaignData, state);

        if (!AvailableCampaignSources.INSTANCE.isUC(dbCampaignData.getSource())) {
            notifyOrderNotificationService
                    .sendNotification(dbCampaignData, updateRequest.getSumUnits(), sum, sumPayed, sumDelta,
                            productRate, isCashbackOnly);
        }
    }

    /**
     * Проверяет все ли изменение баланса целиком прошло за счет бонусов
     * @param updateRequest данные в запросе
     * @param dbCampaignData данные в базе
     * @return true, если пополнение было только бонусами
     */
    private static boolean isCashbackOnly(
            NotifyOrderParameters updateRequest,
            CampaignDataForNotifyOrder dbCampaignData,
            MigrationSchema.State state
    ) {
        // Проверка изменения totalSum (TotalConsumeQty) делается только в том случае, когда это поле влияет на
        // изменение sum в базе в методе NotifyOrderService.notifyOrderInternal
        if (state == MigrationSchema.State.NEW
                && dbCampaignData.getType() == CampaignType.WALLET
                && updateRequest.getTotalSum() != null) {
            var diffTotalSum = updateRequest.getTotalSum()
                    .subtract(nvl(dbCampaignData.getSum(), BigDecimal.ZERO));
            if (diffTotalSum.compareTo(BigDecimal.ZERO) > 0) {
                var diffTotalCashback = updateRequest.getTotalCashback() == null ? BigDecimal.ZERO :
                        updateRequest.getTotalCashback()
                                .subtract(nvl(dbCampaignData.getTotalCashback(), BigDecimal.ZERO));
                if (diffTotalSum.subtract(diffTotalCashback).abs().compareTo(BigDecimal.ONE) >= 0) {
                    return false;
                }
            }
        }
        // Проверка изменения sum_units (ConsumeQty) делается независимо от наличия ОС, как это делается при определении
        // флага sumsChanged в NotifyOrderService.isSumsChanged
        var diffSum = updateRequest.getSumUnits()
                .subtract(BigDecimal.valueOf(nvl(dbCampaignData.getSumUnits(), 0L)));
        if (diffSum.compareTo(BigDecimal.ZERO) > 0) {
            var diffCashback = updateRequest.getCashback() == null ? BigDecimal.ZERO :
                    updateRequest.getCashback()
                            .subtract(nvl(dbCampaignData.getCashback(), BigDecimal.ZERO));
            if (diffSum.subtract(diffCashback).abs().compareTo(BigDecimal.ONE) >= 0) {
                return false;
            }
        }
        logger.info("Automatic pay by cashback only");
        return true;
    }
}
