package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.CampFinishedMailNotification;
import ru.yandex.direct.core.entity.notification.container.NotificationType;
import ru.yandex.direct.core.entity.notification.container.NotifyOrderMailNotification;
import ru.yandex.direct.core.entity.notification.container.NotifyOrderPayType;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.balanceclient.container.CampaignDataForNotifyOrder;

import static ru.yandex.direct.utils.CommonUtils.isValidId;

@ParametersAreNonnullByDefault
@Service
public class NotifyOrderNotificationService {

    private final ShardHelper shardHelper;
    private final UserRepository userRepository;

    private final ClientNdsService clientNdsService;
    private final CampaignService campaignService;
    private final EventLogService eventLogService;
    private final NotificationService notificationService;

    @Autowired
    public NotifyOrderNotificationService(ShardHelper shardHelper,
                                          UserRepository userRepository,
                                          ClientNdsService clientNdsService,
                                          CampaignService campaignService,
                                          EventLogService eventLogService,
                                          NotificationService notificationService) {
        this.shardHelper = shardHelper;
        this.userRepository = userRepository;
        this.clientNdsService = clientNdsService;
        this.campaignService = campaignService;
        this.eventLogService = eventLogService;
        this.notificationService = notificationService;
    }

    /**
     * Отправить уведомление пользователю, если это необходимо
     * Пользователю отправляется письмо. В случае, если он подписался на событие, то и смс
     */
    void sendNotification(CampaignDataForNotifyOrder dbCampaignData, BigDecimal sumUnits, Money sum,
                          Money sumPayed, Money sumDelta, Long productRate, boolean isCashbackOnly) {
        NotifyOrderPayType payType = getPayType(dbCampaignData);

        //о зачислении денег на дубль кампании после копирования письмо не пишем, чтобы не удивлять и не спамить клиента
        if (sumPayed.greaterThanZero() && !dbCampaignData.getFirstAfterCopyConvert()) {
            /*
            Не шлем уведомление по кампании если на ней подключен общий счет и тип кампании из MONEY_TRANSFER
            Условие взято из перлового модуля Notification.pm, метод notification_notify_order_money_in
             */
            if (!(isValidId(dbCampaignData.getWalletId())
                    && CampaignTypeKinds.MONEY_TRANSFER.contains(dbCampaignData.getType()))
                    && !isCashbackOnly) {
                eventLogService.addMoneyInEventLog(dbCampaignData.getCampaignId(), dbCampaignData.getType(), sumPayed,
                        dbCampaignData.getClientId());

                NotifyOrderMailNotification notification =
                        getNotifyOrderMailNotification(NotificationType.NOTIFY_ORDER_MONEY_IN_WO_EVENTLOG,
                                dbCampaignData, sumUnits, sum, sumPayed, productRate, payType);
                notificationService.addNotification(notification);
            }

            if (dbCampaignData.getFinishDate() != null && LocalDate.now().isAfter(dbCampaignData.getFinishDate())) {
                /*
                если на кампанию с прошедшей датой окончания положили деньги (оплатили с задержкой),
                то уведомление об остановке кампании отправляем вместе с письмом о зачислении денег
                (в БК кампания станет активной, но показы не начнутся)
                 */
                eventLogService.addCampFinishedEventLog(dbCampaignData.getCampaignId(), dbCampaignData.getFinishDate(),
                        dbCampaignData.getClientId());

                CampFinishedMailNotification cfNotification = getCampFinishedMailNotification(dbCampaignData);
                notificationService.addNotification(cfNotification);
            }
        } else if (sumDelta.lessThanZero() && payType == NotifyOrderPayType.BLOCKED) {
            /*
            если изменение суммы отрицательное и оплата могла быть только заблокированными деньгами
            то отправляем уведомление, что списали заблокированные деньги
             */
            NotifyOrderMailNotification notification =
                    getNotifyOrderMailNotification(NotificationType.NOTIFY_ORDER_MONEY_OUT_BLOCKING, dbCampaignData,
                            sumUnits, sum, sumPayed, productRate, payType);
            notificationService.addNotification(notification);
        }
    }

    /**
     * Возвращает статус блокировки денег на кампании
     */
    NotifyOrderPayType getPayType(CampaignDataForNotifyOrder dbCampaignData) {
        CampaignForBlockedMoneyCheck campaign = new Campaign()
                .withId(dbCampaignData.getCampaignId())
                .withType(dbCampaignData.getType())
                .withWalletId(dbCampaignData.getWalletId())
                .withUserId(dbCampaignData.getUid())
                .withManagerUserId(dbCampaignData.getManagerUid())
                .withAgencyUserId(dbCampaignData.getAgencyUid())
                .withStatusPostModerate(dbCampaignData.getStatusPostModerate())
                .withStatusModerate(dbCampaignData.getStatusModerate());

        return campaignService.moneyOnCampaignIsBlocked(campaign, false, true)
                ? NotifyOrderPayType.BLOCKED
                : NotifyOrderPayType.ANY;
    }

    /**
     * Возвращает nds агенства если передан agencyId, иначе nds клиента
     *
     * @return НДС как {@link Percent}
     */
    @Nullable
    Percent getAgencyOrClientNds(@Nullable Long agencyId, Long clientId) {
        Long clientIdToFetchNds = (agencyId != null && agencyId > 0) ? agencyId : clientId;
        ClientNds fetchedNds = clientNdsService.getClientNds(ClientId.fromLong(clientIdToFetchNds));
        return fetchedNds == null ? null : fetchedNds.getNds();
    }

    /**
     * Возвращает email агенства
     */
    @Nullable
    String getAgencyEmail(@Nullable Long agencyUid) {
        return agencyUid == null ? null
                : userRepository.getUserEmail(shardHelper.getShardByUserId(agencyUid), agencyUid);
    }

    /**
     * Возвращает количество оплаченных единиц продукта
     * Округляем до трех знаков после запятой т.к. в перле было так:
     * sprintf("%.3f", ($sum_units - $x->{sum_units})/$product->{Rate})
     * Удаляем лишние нули с конца в нецелой части - stripTrailingZeros
     */
    static BigDecimal calcSumPayedUnitsRate(BigDecimal sumUnits, @Nullable Long dbSumUnits, Long productRate) {
        return sumUnits
                .subtract(BigDecimal.valueOf(dbSumUnits == null ? 0 : dbSumUnits))
                .divide(BigDecimal.valueOf(productRate), 3, RoundingMode.HALF_DOWN)
                .stripTrailingZeros();
    }

    /**
     * Есть ли ндс
     */
    static boolean hasNds(CurrencyCode currencyCode, @Nullable Percent nds) {
        return currencyCode != CurrencyCode.YND_FIXED && nds != null && nds.asRatio().compareTo(BigDecimal.ZERO) > 0;
    }

    /**
     * Возвращает заполненую модель для отправки уведомления
     * NOTIFY_ORDER_MONEY_IN_WO_EVENTLOG или NOTIFY_ORDER_MONEY_OUT_BLOCKING
     */
    NotifyOrderMailNotification getNotifyOrderMailNotification(
            NotificationType notificationType,
            CampaignDataForNotifyOrder dbCampaignData, BigDecimal sumUnits, Money sum, Money sumPayed, Long productRate,
            NotifyOrderPayType payType
    ) {
        String agencyEmail = getAgencyEmail(dbCampaignData.getAgencyUid());
        BigDecimal sumPayedUnitsRate = calcSumPayedUnitsRate(sumUnits, dbCampaignData.getSumUnits(), productRate);
        Percent nds = getAgencyOrClientNds(dbCampaignData.getAgencyId(), dbCampaignData.getClientId());

        boolean hasNds = hasNds(sumPayed.getCurrencyCode(), nds);
        Money sumPayedWithoutNds = hasNds ? sumPayed.subtractNds(nds) : sumPayed;

        return new NotifyOrderMailNotification(notificationType)
                .withCampaignId(dbCampaignData.getCampaignId())
                .withCampaignName(dbCampaignData.getName())
                .withCampaignType(dbCampaignData.getType())
                .withWalletId(dbCampaignData.getWalletId())
                .withClientId(dbCampaignData.getClientId())
                .withClientUserId(dbCampaignData.getUid())
                .withClientFullName(dbCampaignData.getFio())
                .withClientLogin(dbCampaignData.getLogin())
                .withClientPhone(dbCampaignData.getPhone())
                .withClientEmail(dbCampaignData.getEmail())
                .withSum(sum)
                .withSumPayed(sumPayedWithoutNds)
                .withSumPayedOriginal(sumPayed)
                .withWithoutNds(hasNds)
                .withNds(nds != null ? nds.asRatio() : null)
                .withSumPayedUnitsRate(sumPayedUnitsRate)
                .withStartTimeTs(dbCampaignData.getStartTimeTs())
                .withStartTimeInFuture(dbCampaignData.getStartTimeInFuture())
                .withAgencyUserId(dbCampaignData.getAgencyUid())
                .withAgencyEmail(agencyEmail)
                .withPayType(payType);
    }

    /**
     * Возвращает заполненую модель для отправки уведомления CAMP_FINISHED_WO_EVENTLOG
     */
    static CampFinishedMailNotification getCampFinishedMailNotification(CampaignDataForNotifyOrder dbCampaignData) {
        return new CampFinishedMailNotification()
                .withCampaignId(dbCampaignData.getCampaignId())
                .withFinishDate(dbCampaignData.getFinishDate())
                .withCampaignName(dbCampaignData.getName())
                .withAgencyUid(dbCampaignData.getAgencyUid())
                .withClientId(dbCampaignData.getClientId())
                .withClientUserId(dbCampaignData.getUid())
                .withClientEmail(dbCampaignData.getEmail())
                .withClientFullName(dbCampaignData.getFio())
                .withClientLogin(dbCampaignData.getLogin())
                .withClientPhone(dbCampaignData.getPhone())
                .withClientLang(dbCampaignData.getLang());
    }
}
