package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.container.LogCampaignBalanceData;
import ru.yandex.direct.common.log.service.LogCampaignBalanceService;
import ru.yandex.direct.core.entity.balanceaggrmigration.lock.AggregateMigrationRedisLockService;
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyOrder;
import ru.yandex.direct.core.entity.campaign.model.CampaignMulticurrencySums;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignsMulticurrencySumsRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.entity.ppcproperty.model.WalletMigrationStateFlag;
import ru.yandex.direct.core.entity.product.model.ProductSimple;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.walletparams.container.WalletParams;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.core.entity.walletparams.service.WalletParamsService;
import ru.yandex.direct.core.entity.xiva.XivaPushesQueueService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbschema.ppc.enums.XivaPushesQueuePushType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.container.CampaignDataForNotifyOrder;
import ru.yandex.direct.intapi.entity.balanceclient.exception.BalanceClientException;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.intapi.entity.balanceclient.repository.NotifyOrderRepository;
import ru.yandex.direct.intapi.entity.balanceclient.service.migration.MigrationSchema;
import ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.currency.Currencies.EPSILON;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.validateCampaignTid;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.validateTotalSum;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class NotifyOrderService {
    static final int CAMPAIGN_NOT_EXISTS_ERROR_CODE = 809;
    static final String CAMPAIGN_DOES_NOT_EXISTS_ERROR_MESSAGE = "Campaign %d does not exists";
    static final String CANT_UPDATE_CAMPAIGNS_MULTICURRENCY_SUMS_ERROR_MESSAGE =
            "Can't update campaigns_multicurrency_sums for campaign %d, new balance_tid is %s";
    static final int GET_LOCK_TROUBLES_ERROR_CODE = 811;
    static final String GET_LOCK_TROUBLES_ERROR_MESSAGE = "Try again later";
    static final int MAX_TIME_MIGRATION_DURATION_SEC = 900; // 15 * 60 == 15 minutes;
    static final long ID_NOT_SET = 0;
    private static final Logger logger = LoggerFactory.getLogger(NotifyOrderService.class);
    private final ShardHelper shardHelper;

    private final NotifyOrderValidationService validationService;
    private final NotifyOrderUpdateCampaignDataService updateCampaignDataService;
    private final NotifyOrderCampaignPostProcessingService postProcessingService;
    private final ProductService productService;
    private final CampaignService campaignService;
    private final LogCampaignBalanceService logCampaignBalanceService;

    private final NotifyOrderRepository notifyOrderRepository;
    private final CampaignRepository campaignRepository;
    private final ClientRepository clientRepository;
    private final CampaignsMulticurrencySumsRepository campaignsMulticurrencySumsRepository;
    private final WalletParamsRepository walletCampaignsRepository;
    private final WalletParamsService walletParamsService;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final AggregateMigrationRedisLockService migrationRedisLockService;

    private final FeatureService featureService;
    private final XivaPushesQueueService xivaPushesQueueService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public NotifyOrderService(
            NotifyOrderValidationService validationService,
            NotifyOrderUpdateCampaignDataService updateCampaignDataService,
            NotifyOrderCampaignPostProcessingService postProcessingService,
            ProductService productService,
            CampaignService campaignService,
            LogCampaignBalanceService logCampaignBalanceService,
            NotifyOrderRepository notifyOrderRepository,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            ClientRepository clientRepository,
            CampaignsMulticurrencySumsRepository campaignsMulticurrencySumsRepository,
            WalletParamsRepository walletCampaignsRepository,
            WalletParamsService walletParamsService,
            PpcPropertiesSupport ppcPropertiesSupport,
            AggregateMigrationRedisLockService migrationRedisLockService,
            FeatureService featureService,
            XivaPushesQueueService xivaPushesQueueService
    ) {
        this.validationService = validationService;
        this.updateCampaignDataService = updateCampaignDataService;
        this.postProcessingService = postProcessingService;
        this.productService = productService;
        this.campaignService = campaignService;
        this.clientRepository = clientRepository;
        this.logCampaignBalanceService = logCampaignBalanceService;
        this.shardHelper = shardHelper;
        this.notifyOrderRepository = notifyOrderRepository;
        this.campaignRepository = campaignRepository;
        this.campaignsMulticurrencySumsRepository = campaignsMulticurrencySumsRepository;
        this.walletCampaignsRepository = walletCampaignsRepository;
        this.walletParamsService = walletParamsService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.migrationRedisLockService = migrationRedisLockService;
        this.featureService = featureService;
        this.xivaPushesQueueService = xivaPushesQueueService;
    }

    /**
     * Нужно ли обновлять мультивалютные данные для кампаний
     */
    static boolean campaignsMulticurrencySumsShouldBeUpdated(CampaignDataForNotifyOrder dbCampaignData,
                                                             CampaignMulticurrencySums campaignMulticurrencySums) {
        BigDecimal cmsSum = nvl(dbCampaignData.getCmsSum(), BigDecimal.ZERO);
        BigDecimal cmsChipsCost = nvl(dbCampaignData.getCmsChipsCost(), BigDecimal.ZERO);
        BigDecimal cmsChipsSpent = nvl(dbCampaignData.getCmsChipsSpent(), BigDecimal.ZERO);

        return cmsSum.compareTo(campaignMulticurrencySums.getSum()) != 0
                || cmsChipsCost.compareTo(campaignMulticurrencySums.getChipsCost()) != 0
                || cmsChipsSpent.compareTo(campaignMulticurrencySums.getChipsSpent()) != 0;
    }

    /**
     * Обновлять сумму общих фишек нужно для кампаний под общим счётом и новой схемы обработки нотификаций
     *
     * @return true/false нужно ли обновлять
     */
    static boolean walletTotalSpentShouldBeUpdated(CampaignDataForNotifyOrder dbCampaignData,
                                                   NotifyOrderParameters updateRequest,
                                                   MigrationSchema.State state) {
        BigDecimal chipsCost = nvl(updateRequest.getChipsCost(), BigDecimal.ZERO);
        BigDecimal cmsChipsCost = nvl(dbCampaignData.getCmsChipsCost(), BigDecimal.ZERO);

        return state == MigrationSchema.State.NEW
                && dbCampaignData.getWalletId() > 0
                && chipsCost.compareTo(cmsChipsCost) != 0;
    }

    public BalanceClientResponse notifyOrder(NotifyOrderParameters updateRequest) {
        BalanceClientResponse validationResponse;
        validationResponse = validationService.validateRequest(updateRequest);
        if (validationResponse != null) {
            return validationResponse;
        }

        int campaignShard = shardHelper.getShardByCampaignId(updateRequest.getCampaignId());

        CampaignDataForNotifyOrder dbCampaignData =
                notifyOrderRepository.fetchCampaignData(campaignShard, updateRequest.getCampaignId());

        validationResponse = validate(dbCampaignData, updateRequest);
        if (validationResponse != null) {
            return validationResponse;
        }

        MigrationSchema schema = processMigrationSchema(campaignShard, dbCampaignData);
        if (schema.getState() == MigrationSchema.State.LOCK_TROUBLES) {
            return BalanceClientResponse.error(GET_LOCK_TROUBLES_ERROR_CODE, GET_LOCK_TROUBLES_ERROR_MESSAGE);
        }

        try {
            return notifyOrderInternal(campaignShard, dbCampaignData, updateRequest, schema.getState());
        } finally {
            if (schema.getState() == MigrationSchema.State.OLD_WITH_LOCK) {
                boolean isUnlocked = migrationRedisLockService.unlock(schema.getLock());
                if (!isUnlocked) {
                    logger.warn("Cannot release lock on migration status {} for campaign {}",
                            schema.getState().name(), updateRequest.getCampaignId());
                }
            }
        }
    }

    private BalanceClientResponse validate(
            @Nullable CampaignDataForNotifyOrder dbCampaignData,
            NotifyOrderParameters updateRequest
    ) {
        BalanceClientResponse validationResponse;

        if (dbCampaignData == null) {
            if (updateRequest.hasAnyMoney()) {
                String message = String.format(CAMPAIGN_DOES_NOT_EXISTS_ERROR_MESSAGE, updateRequest.getCampaignId());
                logger.warn(message);
                return BalanceClientResponse.error(CAMPAIGN_NOT_EXISTS_ERROR_CODE, message);
            } else {
                logger.warn("Accept notification without money for not existing campaign {}",
                        updateRequest.getCampaignId());
                return BalanceClientResponse.success();
            }
        }

        validationResponse = validationService.validateCampaignType(updateRequest.getServiceId(),
                dbCampaignData.getType(), dbCampaignData.getCampaignId());
        if (validationResponse != null) {
            return validationResponse;
        }

        validationResponse = validateCampaignTid(updateRequest, dbCampaignData.getBalanceTid());
        if (validationResponse != null) {
            return validationResponse;
        }

        validationResponse = validationService.validateClientCurrencyConversionState(dbCampaignData.getClientId());
        if (validationResponse != null) {
            return validationResponse;
        }

        return null;
    }

    private void sendPush(ClientId clientId) {
        xivaPushesQueueService.addPushToQueue(clientId, XivaPushesQueuePushType.TOTAL_SUM_CHANGED);
    }

    @SuppressWarnings("checkstyle:methodlength")
    private BalanceClientResponse notifyOrderInternal(int campaignShard, CampaignDataForNotifyOrder dbCampaignData,
                                                      NotifyOrderParameters updateRequest,
                                                      MigrationSchema.State state) {
        BalanceClientResponse validationResponse;

        boolean multicurrencySumsUpdated = updateMulticurrencySums(campaignShard, dbCampaignData, updateRequest, state);

        ProductSimple productInfo = productService.getProductById(dbCampaignData.getProductId());

        Money sum;
        if (isCampaignModifyConverted(dbCampaignData.getCurrency(), dbCampaignData.getCurrencyConverted())) {
            // Сконвертированная без копирования кампания.
            // В этом случае в Балансе остаётся старый фишечный продукт, но все оплаты идут в рублях.
            //
            // Количество фишечного товара в этом случае использовать нельзя,
            // поэтому смотрим на специальное поле ConsumeMoneyQty
            // sumRealMoney в этом случае всегда присылается Балансом, поэтому явной проверки на null тут нет, есть
            // checkNotNull в Money
            sum = Money.valueOf(updateRequest.getSumRealMoney(), CurrencyCode.RUB);
        } else {
            // Для всех остальных заказов смотрим на количество зачисленного товара,
            // будь это фишки или "рубли/доллары/евро/etc директа".
            //
            // ProductCurrency должна всегда** приходить для директовских кампаний
            //
            // ** подробнее в коде validateProductCurrency
            validationResponse = validationService
                    .validateProductCurrency(updateRequest.getServiceId(), updateRequest.getProductCurrency(),
                            productInfo.getCurrencyCode(), updateRequest.getCampaignId());
            if (validationResponse != null) {
                return validationResponse;
            }

            sum = Money.valueOf(productInfo.getPrice(), productInfo.getCurrencyCode())
                    .multiply(updateRequest.getSumUnits())
                    .divide(productInfo.getRate());
        }

        validationResponse = validationService
                .validateSumOnEmptyCampaign(dbCampaignData.getStatusEmpty(), sum, dbCampaignData.getCampaignId());
        if (validationResponse != null) {
            return validationResponse;
        }

        BigDecimal dbSumBalance = dbCampaignData.getSumBalance();
        BigDecimal sumBalance = BigDecimal.ZERO;

        if (state == MigrationSchema.State.NEW) {
            // в поле campaigns.sumBalance будем хранить старое значение sum на случай отката.
            sumBalance = sum.bigDecimalValue();

            // значение sum пересчитываем в зависимости от типа кампании (ОС или нет).
            BigDecimal value;
            if (dbCampaignData.getType() == CampaignType.WALLET) {
                if (updateRequest.getTotalSum() != null) {
                    value = updateRequest.getTotalSum();
                } else {
                    // если клиент оплатил ОС до того, как создал под ним кампанию, ОС не считается
                    // групповым заказом в Биллинге, и TotalConsumeQty не приходит. В этом случае
                    // вычисляем сумму так же, как для обычных кампаний
                    value = sum.bigDecimalValue();
                }
            } else {
                value = BigDecimal.ZERO;
            }
            sum = Money.valueOf(value, sum.getCurrencyCode());
        }

        Money dbSum = Money.valueOf(dbCampaignData.getSum(), dbCampaignData.getCurrency());
        Money sumDelta = sum.subtract(dbSum);

        addLogBalance(dbCampaignData, sum, sumDelta, sumBalance, updateRequest.getTid());

        Money sumSpent = Money.valueOf(dbCampaignData.getSumSpent(), dbCampaignData.getCurrency());

        List<CampaignForNotifyOrder> campsInWallet;
        if (dbCampaignData.getType() == CampaignType.WALLET) {
            campsInWallet = campaignRepository
                    .getCampaignsForNotifyOrder(campaignShard, dbCampaignData.getUid(), updateRequest.getCampaignId());

            // учитываем ещё не покрытые перетраты на заказах под общим счётом
            sumSpent = sumSpent.add(calcUncoveredSpents(campsInWallet, dbCampaignData.getCurrency()));
        } else {
            campsInWallet = emptyList();
        }

        Money sumRestNew = sum.subtract(sumSpent);
        Money sumRestOld = dbSum.subtract(sumSpent);

        boolean campaignShouldBeUnarchived = dbCampaignData.getArchived()
                && dbCampaignData.getWalletId() == ID_NOT_SET
                && sumRestNew.greaterThanOrEqualEpsilon();
        if (campaignShouldBeUnarchived) {
            logger.info("Unarchiving campaign {}", dbCampaignData.getCampaignId());
            campaignService.unarchiveCampaign(dbCampaignData.getUid(), dbCampaignData.getCampaignId());
        }

        boolean sumsChanged = isSumsChanged(sum, dbSum, updateRequest.getSumUnits(), dbCampaignData.getSumUnits());
        if (state == MigrationSchema.State.NEW) {
            // для новой схемы учёта нотификаций дополнительно проверяется изменение sumBalance
            sumsChanged = sumsChanged || sumBalance.compareTo(dbSumBalance) != 0;
        }

        boolean isCampaignChanged = updateCampaignDataService.updateCampaignData(
                campaignShard, sumsChanged, sum, sumDelta, sumBalance, dbCampaignData, updateRequest);

        // Изменилась только стоимость/количество фишек и больше ничего
        if (multicurrencySumsUpdated && !isCampaignChanged) {
            postProcessingService.processUnchangedCampaign(campaignShard, updateRequest.getCampaignId());
        }

        // Денег на кампании не было, и вот они поступили
        if (sumRestOld.lessThanOrEqualEpsilon() && sumRestNew.greaterThanOrEqualEpsilon()) {
            postProcessingService.processMoneyRefillFromZero(campaignShard, dbCampaignData, campsInWallet);
        } else if (!campsInWallet.isEmpty()) {
            // проверяем, не поступили ли деньги, с учётом автоовердрафта
            // если если поступили - переотправляем кампании под кошельком в БК для рестарта автобюджета
            // TODO: вероятно овердрафт нужно учитывать и в верхнем случае
            var clientAutoOverdraftInfo = clientRepository.getClientsAutoOverdraftInfo(
                    campaignShard, List.of(ClientId.fromLong(dbCampaignData.getClientId()))
            ).stream().findFirst().orElse(null);
            if (clientAutoOverdraftInfo != null) {
                try {
                    var autoOverdraftAddition = AutoOverdraftUtils.calculateAutoOverdraftAddition(
                            dbCampaignData.getCurrency(), sum.bigDecimalValue(), sumSpent.bigDecimalValue(),
                            clientAutoOverdraftInfo);
                    var wasMoney = sumRestOld.bigDecimalValue().add(autoOverdraftAddition).compareTo(EPSILON) > 0;
                    var hasMoney = sumRestNew.bigDecimalValue().add(autoOverdraftAddition).compareTo(EPSILON) > 0;
                    if (wasMoney != hasMoney) {
                        logger.info("Resend to BS campaigns, binded to {} due to refill (consider autooverdraft)",
                                dbCampaignData.getCampaignId());
                        var campaignIds = mapList(campsInWallet, CampaignForNotifyOrder::getId);
                        campaignRepository.resetBannerSystemSyncStatus(campaignShard, campaignIds);
                    }
                } catch (RuntimeException e) {
                    logger.warn("Skip reset statusBsSync due to error", e);
                }
            }
        }

        // Зачисленная сумма изменилась
        if (sumsChanged) {
            postProcessingService
                    .processSumOnCampChange(campaignShard, updateRequest, dbCampaignData, productInfo.getRate(), sum,
                            sumDelta, state);
        }

        // для кошельков (у которых валюта совпадает с валютой клиента) сохраняем общие суммы по кошельку и всем
        // кампаниям под ним
        boolean walletParamsShouldBeUpdated = dbCampaignData.getType() == CampaignType.WALLET
                && dbCampaignData.getCurrency() == dbCampaignData.getClientWorkCurrency()
                && updateRequest.getTid() >= nvl(dbCampaignData.getTotalBalanceTid(), 0L);
        if (walletParamsShouldBeUpdated && updateRequest.getTotalSum() == null && !updateRequest.hasAnyMoney()) {
            // DIRECT-78810: NotifyOrder: не падать при отсутствии TotalConsumeQty в нотификациях без денег
            logger.warn("Skip processing {} for campaign {} - notification does not contain money",
                    NotifyOrderParameters.TOTAL_SUM_FIELD_NAME, updateRequest.getCampaignId());
            walletParamsShouldBeUpdated = false;
        }
        if (walletParamsShouldBeUpdated) {
            validationResponse = validateTotalSum(updateRequest.getTotalSum());
            if (validationResponse != null) {
                return validationResponse;
            }

            boolean isTotalSumChanged = updateRequest.getTotalSum()
                    .subtract(nvl(dbCampaignData.getTotalSum(), BigDecimal.ZERO), Money.MONEY_MATH_CONTEXT)
                    .abs(Money.MONEY_MATH_CONTEXT)
                    .compareTo(EPSILON) > 0;
            if (isTotalSumChanged) {
                // делаем запросы в БД только если сумма изменились, в противном случае будет непонятно - проблема в
                // изменившемся tid или в неизменившейся сумме
                // в лог нотификаций с общими суммами по кошелькам пишем только при успешном сохранении нотификации в БД
                WalletParams walletCampaign = new WalletParams()
                        .withTotalSum(updateRequest.getTotalSum())
                        .withTotalBalanceTid(updateRequest.getTid())
                        .withWalletId(dbCampaignData.getCampaignId());
                updateWalletParams(campaignShard, dbCampaignData.getTotalBalanceTid(), walletCampaign);

                ClientId clientId = ClientId.fromLong(dbCampaignData.getClientId());
                if (featureService.isEnabledForClientId(clientId, FeatureName.SEND_XIVA_PUSHES) &&
                        featureService.isEnabledForClientId(clientId, FeatureName.SEND_PUSH_TOTAL_SUM_CHANGED)) {
                    sendPush(clientId);
                }
            }
        }

        return BalanceClientResponse.success();
    }

    /**
     * Вычисление схемы обработки нотификаций из биллинга. Возможные состояния описаны в {@code MigrationSchema}
     */
    MigrationSchema processMigrationSchema(int shard, CampaignDataForNotifyOrder dbCampaignData) {
        boolean isCampaignTypeWallet = dbCampaignData.getType() == CampaignType.WALLET;
        boolean isCampaignUnderWallet = dbCampaignData.getWalletId() > 0;

        if (!isCampaignTypeWallet && !isCampaignUnderWallet) {
            return MigrationSchema.build(MigrationSchema.State.OLD);
        }

        WalletMigrationStateFlag migrationInfoFlag = readPpcPropertyWalletMigrationStateFlag();

        AggregatingSumStatus migrationStatus = dbCampaignData.getWalletAggregateMigrated();
        boolean isCampaignCurrencyNotChips = dbCampaignData.getCurrency() != CurrencyCode.YND_FIXED;
        boolean lastMigrationFlagChangedLessThanMaxDuration = MAX_TIME_MIGRATION_DURATION_SEC >
                TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()) - nvl(migrationInfoFlag.getTime(), 0L);

        if (migrationStatus == AggregatingSumStatus.YES) {
            return MigrationSchema.build(MigrationSchema.State.NEW);
        } else if (migrationStatus == AggregatingSumStatus.NO
                && isCampaignCurrencyNotChips
                && (migrationInfoFlag.isEnabled() || lastMigrationFlagChangedLessThanMaxDuration)) {
            // лок берётся на номер общего счёта.
            Long walletId = dbCampaignData.getType() == CampaignType.WALLET
                    ? dbCampaignData.getCampaignId()
                    : dbCampaignData.getWalletId();

            DistributedLock lock = migrationRedisLockService.lock(walletId);
            if (!lock.isLocked()) {
                logger.info("[LOCK] cant acquire migration lock: result getting lock is false after trying");
                return MigrationSchema.build(MigrationSchema.State.LOCK_TROUBLES);
            }

            dbCampaignData = notifyOrderRepository.fetchCampaignData(shard, dbCampaignData.getCampaignId());
            migrationStatus = dbCampaignData.getWalletAggregateMigrated();
            if (migrationStatus == AggregatingSumStatus.YES) {
                boolean isUnlocked = migrationRedisLockService.unlock(lock);
                if (isUnlocked) {
                    return MigrationSchema.build(MigrationSchema.State.NEW);
                } else {
                    logger.warn("Cannot release migration lock on migration status {} for campaign {}",
                            migrationStatus, dbCampaignData.getCampaignId());
                    return MigrationSchema.build(MigrationSchema.State.LOCK_TROUBLES);
                }
            } else {
                return MigrationSchema.build(MigrationSchema.State.OLD_WITH_LOCK).withLock(lock);
            }
        }

        return MigrationSchema.build(MigrationSchema.State.OLD);
    }

    private WalletMigrationStateFlag readPpcPropertyWalletMigrationStateFlag() {
        String propertyName = PpcPropertyEnum.WALLET_SUMS_MIGRATION_STATE.getName();

        return ppcPropertiesSupport.find(propertyName)
                .map(prop -> {
                    try {
                        return JsonUtils.fromJson(prop, new TypeReference<WalletMigrationStateFlag>() {
                        });
                    } catch (Exception e) {
                        logger.error("Failed to parse json array property " + propertyName, e);
                        return WalletMigrationStateFlag.disabled();
                    }
                })
                .orElse(WalletMigrationStateFlag.disabled());
    }

    /**
     * Сконвертированна ли кампания без копирования
     */
    boolean isCampaignModifyConverted(CurrencyCode currencyCode, @Nullable Boolean currencyConverted) {
        return currencyCode == CurrencyCode.RUB
                && currencyConverted != null && currencyConverted;
    }

    boolean isSumsChanged(Money sum, Money dbSum, BigDecimal sumUnits, @Nullable Long dbSumUnits) {
        return dbSum.compareTo(sum) != 0
                || sumUnits.compareTo(BigDecimal.valueOf(dbSumUnits == null ? 0 : dbSumUnits)) != 0;
    }

    /**
     * Сохраняем логи в файл для загрузки в CH
     */
    void addLogBalance(CampaignDataForNotifyOrder campaign, Money sum, Money sumDelta, BigDecimal sumBalance,
                       long balanceTid) {
        LogCampaignBalanceData logCampaignBalanceData = new LogCampaignBalanceData()
                .withCid(campaign.getCampaignId())
                .withType(campaign.getType().name().toLowerCase())
                .withCurrency(campaign.getCurrency().name())
                .withClientId(campaign.getClientId())
                .withTid(balanceTid)
                .withSum(sum.bigDecimalValue())
                .withSumBalance(sumBalance)
                .withSumDelta(sumDelta.bigDecimalValue());
        logCampaignBalanceService.logCampaignBalance(logCampaignBalanceData);
    }

    void updateWalletParams(int campaignShard, @Nullable Long currentWalletTid, WalletParams walletParams) {
        if (currentWalletTid != null) {
            //запись в таблице уже есть, обновляем только при условии, что кто-то раньше не обновил (смотрим на
            // total_balance_tid)
            boolean walletWasUpdated =
                    walletCampaignsRepository.updateWalletParams(campaignShard, currentWalletTid, walletParams);
            if (!walletWasUpdated) {
                logger.warn("Can't update wallet_campaigns for campaign {}, new balance_tid is {}",
                        walletParams.getWalletId(), walletParams.getTotalBalanceTid());
            }
        } else {
            // записи в таблице нет, вставляем
            // если кто-то уже успел вставить запись, упадём на дублирующемся первичном ключе
            walletCampaignsRepository.addWalletParams(campaignShard, walletParams);
        }
    }

    /**
     * Возвращает не покрытые перетраты на заказах под общим счётом
     */
    Money calcUncoveredSpents(List<CampaignForNotifyOrder> campsInWallet, CurrencyCode currency) {
        return campsInWallet.stream()
                .map(camp -> camp.getSum().subtract(camp.getSumSpent(), Money.MONEY_MATH_CONTEXT))
                .map(total -> Money.valueOf(total, currency))
                .filter(Money::lessThanZero)
                .map(Money::abs)
                .reduce(Money.valueOf(BigDecimal.ZERO, currency), Money::add);
    }

    /**
     * Добавление или обновление мультивалютных данных для кампаний,
     * переходящих в реальную валюту без остановки кампаний
     *
     * @return была ли добавлена или обновлена запись
     */
    boolean updateMulticurrencySums(int shard, CampaignDataForNotifyOrder dbCampaignData,
                                    NotifyOrderParameters updateRequest, MigrationSchema.State state) {
        if (dbCampaignData.getCurrency() == CurrencyCode.YND_FIXED
                || isCampaignModifyConverted(dbCampaignData.getCurrency(), dbCampaignData.getCurrencyConverted())) {
            CampaignMulticurrencySums campaignMulticurrencySums = new CampaignMulticurrencySums()
                    .withId(dbCampaignData.getCampaignId())
                    .withSum(nvl(updateRequest.getSumRealMoney(), BigDecimal.ZERO))
                    .withChipsCost(nvl(updateRequest.getChipsCost(), BigDecimal.ZERO))
                    .withChipsSpent(nvl(updateRequest.getChipsSpent(), BigDecimal.ZERO))
                    .withAvgDiscount(BigDecimal.ZERO)
                    .withBalanceTid(updateRequest.getTid());
            if (campaignsMulticurrencySumsShouldBeUpdated(dbCampaignData, campaignMulticurrencySums)) {
                if (dbCampaignData.getCmsBalanceTid() != null) {
                    boolean sumsWasUpdated = campaignsMulticurrencySumsRepository
                            .updateCampaignsMulticurrencySumsByCidAndBalanceTid(shard, campaignMulticurrencySums,
                                    dbCampaignData.getCmsBalanceTid());
                    if (!sumsWasUpdated) {
                        String message = String.format(CANT_UPDATE_CAMPAIGNS_MULTICURRENCY_SUMS_ERROR_MESSAGE,
                                dbCampaignData.getCampaignId(), updateRequest.getTid());
                        logger.warn(message);
                        throw new BalanceClientException(message);
                    }
                } else {
                    campaignsMulticurrencySumsRepository
                            .insertCampaignsMulticurrencySums(shard, campaignMulticurrencySums);
                }

                if (walletTotalSpentShouldBeUpdated(dbCampaignData, updateRequest, state)) {
                    walletParamsService.updateTotalCost(dbCampaignData.getWalletId());
                }

                return true;
            }
        }
        return false;
    }
}
