package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StatusMail;
import ru.yandex.direct.core.entity.campaignpayment.repository.CampPaymentsInfoRepository;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.entity.balanceclient.container.CampaignDataForNotifyOrder;
import ru.yandex.direct.intapi.entity.balanceclient.container.NotifyOrderDbCampaignChanges;
import ru.yandex.direct.intapi.entity.balanceclient.exception.BalanceClientException;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.intapi.entity.balanceclient.repository.NotifyOrderRepository;

import static ru.yandex.direct.core.entity.campaign.MediaCampaignUtil.calcMediaSumSpent;

@ParametersAreNonnullByDefault
@Service
public class NotifyOrderUpdateCampaignDataService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyOrderUpdateCampaignDataService.class);

    static final String CANT_UPDATE_CAMPAIGN_ERROR_MESSAGE =
            "Can't update campaign %d, new balance_tid is %s, wallet_cid is %s";

    static final Integer PAID_BY_CERTIFICATE = 1;

    private final NotifyOrderRepository notifyOrderRepository;
    private final DslContextProvider dslContextProvider;
    private final CampPaymentsInfoRepository campPaymentsInfoRepository;
    private final int bayanServiceId;

    public NotifyOrderUpdateCampaignDataService(
            NotifyOrderRepository notifyOrderRepository,
            DslContextProvider dslContextProvider,
            CampPaymentsInfoRepository campPaymentsInfoRepository,
            @Value("${balance.bayanServiceId}") int bayanServiceId) {
        this.notifyOrderRepository = notifyOrderRepository;
        this.dslContextProvider = dslContextProvider;
        this.campPaymentsInfoRepository = campPaymentsInfoRepository;
        this.bayanServiceId = bayanServiceId;
    }

    /**
     * Обновление параметров кампании и добавление/обновление информации о платеже на кампанию одной транзакцией
     *
     * @return были ли изменения в параметрах кампании. Обновление balance_tid не считается за изменением
     */
    boolean updateCampaignData(int campaignShard, boolean sumsChanged, Money sum, Money sumDelta, BigDecimal sumBalance,
                               CampaignDataForNotifyOrder dbCampaignData, NotifyOrderParameters updateRequest) {
        NotifyOrderDbCampaignChanges campaignChanges = new NotifyOrderDbCampaignChanges();
        boolean isCampaignChanged = false;
        if (sumsChanged) {
            isCampaignChanged = true;
            campaignChanges.setSum(sum.bigDecimalValue());
            Long sumUnits = getSumUnitsByServiceId(updateRequest.getServiceId(), updateRequest.getSumUnits());
            campaignChanges.setSumUnits(sumUnits);

            campaignChanges.setSumBalance(sumBalance);

            BigDecimal minPay = Currencies.getCurrency(dbCampaignData.getCurrency()).getMinSumInterpreteAsPayment();
            if (sumDelta.greaterThanZero() && sumDelta.bigDecimalValue().compareTo(minPay) >= 0) {
                campaignChanges.setSumLast(sumDelta.bigDecimalValue());
                campaignChanges.setStatusMail(StatusMail.NOT_SENT);
                campaignChanges.setSumToPay(BigDecimal.ZERO);
            }

            if (dbCampaignData.getType() == CampaignType.MCB) {
                Long dbSumSpentUnits = dbCampaignData.getSumSpentUnits();
                dbSumSpentUnits = dbSumSpentUnits == null ? 0L : dbSumSpentUnits;
                // для Баяна рассчитываем фиктивную истраченную в деньгах сумму пропорционально использованным показам
                campaignChanges.setSumSpent(calcMediaSumSpent(sum, dbSumSpentUnits, sumUnits));
            }
        }

        if (updateRequest.getCashback() != null &&
                !updateRequest.getCashback().equals(dbCampaignData.getCashback())) {
            campaignChanges.setCashback(updateRequest.getCashback());
        }

        if (updateRequest.getTotalCashback() != null &&
                !updateRequest.getTotalCashback().equals(dbCampaignData.getTotalCashback())) {
            campaignChanges.setTotalCashback(updateRequest.getTotalCashback());
        }

        if (!dbCampaignData.getBalanceTid().equals(updateRequest.getTid())) {
            campaignChanges.setBalanceTid(updateRequest.getTid());
        }

        if (PAID_BY_CERTIFICATE.equals(updateRequest.getPaidByCertificate())) {
            isCampaignChanged = true;
            campaignChanges.setPaidByCertificate();
        }

        if (campaignChanges.isChanged()) {
            if (isCampaignChanged) {
                campaignChanges.resetStatusBsSynced();
            }

            updateCampaignDataInTransaction(campaignShard, updateRequest.getCampaignId(), updateRequest.getTid(),
                    dbCampaignData.getBalanceTid(), dbCampaignData.getWalletId(), campaignChanges, sumDelta);
        }
        return isCampaignChanged;
    }

    /**
     * Обновление параметров кампании и добавление/обновление информации о платеже на кампанию одной транзакцией
     *
     * @throws BalanceClientException если кто-то успел обновить tid, или подключить ОС до нас
     */
    void updateCampaignDataInTransaction(
            int campaignShard, Long campaignId, Long balanceTid, Long campaignBalanceTid,
            Long campaignWalletId, NotifyOrderDbCampaignChanges campaignChanges, Money sumDelta
    ) throws BalanceClientException {
        dslContextProvider.ppcTransaction(campaignShard, configuration -> {
            DSLContext txContext = DSL.using(configuration);

            boolean campaignWasUpdated = notifyOrderRepository
                    .updateCampaignData(txContext, campaignId, campaignBalanceTid, campaignWalletId, campaignChanges);

            if (!campaignWasUpdated) {
                // умираем, если кто-то успел обновить tid или включить ОС до нас, после того, как мы делали select
                String message =
                        String.format(CANT_UPDATE_CAMPAIGN_ERROR_MESSAGE, campaignId, balanceTid, campaignWalletId);
                logger.warn(message);
                throw new BalanceClientException(message);
            }

            if (sumDelta.greaterThanZero()) {
                campPaymentsInfoRepository
                        .addCampaignPayment(txContext, campaignId, sumDelta.bigDecimalValue());
            }
        });
    }

    Long getSumUnitsByServiceId(int serviceId, BigDecimal sumUnits) {
        if (serviceId == bayanServiceId) {
            // Баланс присылает только целые числа для Баяновских кампаний
            return sumUnits.longValueExact();
        } else {
            // т.к. для Директовсих кампаний фишечное зачисление не имеет смысла - берем неточное значение
            return sumUnits.longValue();
        }
    }
}
