package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.NotifyPromoOrderMailNotification;
import ru.yandex.direct.core.entity.yandexagencyorder.model.Status;
import ru.yandex.direct.core.entity.yandexagencyorder.model.YandexAgencyOrder;
import ru.yandex.direct.core.entity.yandexagencyorder.repository.YandexAgencyOrdersRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class NotifyPromoOrderService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyPromoOrderService.class);

    private final YandexAgencyOrdersRepository yandexAgencyOrdersRepository;
    private final ShardHelper shardHelper;
    private final NotifyPromoOrderValidationService notifyPromoOrderValidationService;
    private final NotificationService notificationService;

    @Autowired
    public NotifyPromoOrderService(
            YandexAgencyOrdersRepository yandexAgencyOrdersRepository, ShardHelper shardHelper,
            NotifyPromoOrderValidationService notifyPromoOrderValidationService,
            NotificationService notificationService) {
        this.yandexAgencyOrdersRepository = yandexAgencyOrdersRepository;
        this.shardHelper = shardHelper;
        this.notifyPromoOrderValidationService = notifyPromoOrderValidationService;
        this.notificationService = notificationService;
    }

    public BalanceClientResponse notifyPromoOrder(List<NotifyOrderParameters> notifyOrderParametersList) {

        ValidationResult<List<NotifyOrderParameters>, Defect>
                preValidationResult = notifyPromoOrderValidationService.preValidate(notifyOrderParametersList);

        if (preValidationResult.hasAnyErrors()) {
            logger.error("Can't process notifyPromoOrder request. Errors: {}", preValidationResult.flattenErrors());
            return BalanceClientResponse.criticalError("Invalid data");
        }

        List<YandexAgencyOrder> yandexAgencyOrderList = mapList(notifyOrderParametersList,
                notifyParam -> new YandexAgencyOrder()
                        //todo ssdmitriev: на самом деле это не CampaignId, надо будет изменить название.
                        .withId(notifyParam.getCampaignId())
                        .withYaOrderStatus(Status.PAID));

        List<Long> yandexAgencyOrderIds = mapList(yandexAgencyOrderList, YandexAgencyOrder::getId);

        List<YandexAgencyOrder> currentNotifyOrders = StreamEx.of(shardHelper.dbShards()).map(shard ->
                yandexAgencyOrdersRepository.getYandexAgencyOrdersByOrderId(
                        shard, yandexAgencyOrderIds))
                .flatMap(Collection::stream)
                .toList();

        ValidationResult<List<Long>, Defect> validationResult =
                notifyPromoOrderValidationService.validateOrderIdsExist(yandexAgencyOrderIds, currentNotifyOrders);

        if (validationResult.hasAnyErrors()) {
            logger.warn("Ids of some Orders do not exist: {}", validationResult.flattenErrors());
            return BalanceClientResponse.criticalError("Order does not exist");
        }

        ValidationResult<List<YandexAgencyOrder>, Defect> statusValidationResult =
                notifyPromoOrderValidationService.validateOrdersAreNotCompleted(currentNotifyOrders);

        if (statusValidationResult.hasAnyErrors()) {
            logger.warn("Statuses of some orders differ from NEW. Errors: {}", statusValidationResult.flattenErrors());
            return BalanceClientResponse.criticalError("Order has already paid");
        }

        //обновляем статус на Paid, если заказ не был переотправлен или на Completed, если был
        //для клиента может существовать всего один заказ
        Map<Long, YandexAgencyOrder> currentNotifyOrdersById = listToMap(
                currentNotifyOrders,
                YandexAgencyOrder::getId, Function.identity()
        );

        LocalDateTime now = LocalDateTime.now();

        Function<YandexAgencyOrder, Status> calculateStatus =
                order -> currentNotifyOrdersById.get(order.getId()).getYaOrderStatus()
                        .equals(Status.RESURRECTED) ? Status.COMPLETED : Status.PAID;
        yandexAgencyOrderList.forEach(order -> order.setYaOrderStatus(calculateStatus.apply(order)));

        // Изменения yandexAgencyOrderList yaOrderStatus и lastChange
        Map<Long, ModelChanges<YandexAgencyOrder>> modelChangesById = StreamEx.of(yandexAgencyOrderList)
                .map(newNotifyOrder -> new ModelChanges<>(newNotifyOrder.getId(), YandexAgencyOrder.class)
                        .process(newNotifyOrder.getYaOrderStatus(), YandexAgencyOrder.YA_ORDER_STATUS)
                        .process(now, YandexAgencyOrder.LAST_CHANGE))
                .toMap(ModelChanges::getId, Function.identity());

        // Примененнные изменения yandexAgencyOrderList
        Map<Long, AppliedChanges<YandexAgencyOrder>> appliedChangesByClientIds =
                EntryStream.of(currentNotifyOrdersById)
                        .filterValues(agencyOrder -> modelChangesById.containsKey(agencyOrder.getId()))
                        .mapKeys(currentNotifyOrdersById::get)
                        .mapKeys(YandexAgencyOrder::getClientId)
                        .mapValues(agencyOrder -> modelChangesById.get(agencyOrder.getId()).applyTo(agencyOrder))
                        .toMap();


        shardHelper.groupByShard(
                mapList(currentNotifyOrdersById.values(), YandexAgencyOrder::getClientId),
                ShardKey.CLIENT_ID
        ).stream()
                .mapValues(clientIds -> mapList(clientIds, appliedChangesByClientIds::get))
                .forKeyValue(yandexAgencyOrdersRepository::updateYandexOrder);

        // После применения изменений в объектах модели хранятся новые значения
        sendNotifications(filterList(
                currentNotifyOrdersById.values(),
                order -> !order.getYaOrderStatus().equals(Status.COMPLETED)));

        return BalanceClientResponse.success();
    }

    /**
     * Рассылка уведомлений об успешной оплате услуг Я.Агенства
     */
    private void sendNotifications(Collection<YandexAgencyOrder> orders) {
        StreamEx.of(orders)
                .map(this::successNotification)
                .forEach(notificationService::addNotification);
    }

    private NotifyPromoOrderMailNotification successNotification(YandexAgencyOrder order) {
        Long clientId = order.getClientId();
        int numericProductType = order.getProductType().intValue();
        return NotifyPromoOrderMailNotification.success(clientId, numericProductType);
    }
}
