package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.yandexagencyorder.model.Status;
import ru.yandex.direct.core.entity.yandexagencyorder.model.YandexAgencyOrder;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
class NotifyPromoOrderValidationService {
    ValidationResult<List<NotifyOrderParameters>, Defect> preValidate(
            List<NotifyOrderParameters> notifyOrderParametersList) {
        return ListValidationBuilder.of(notifyOrderParametersList, Defect.class)
                .check(notNull())
                .check(eachNotNull())
                .check(maxListSize(1))
                .getResult();
    }

    /**
     * Проверяем, что id заказов существуют
     */
    ValidationResult<List<Long>, Defect> validateOrderIdsExist(List<Long> orderIds,
                                                               List<YandexAgencyOrder> currentYandexOrders) {
        Set<Long> currentYandexOrdersIds = StreamEx.of(currentYandexOrders).map(YandexAgencyOrder::getId).toSet();
        return ListValidationBuilder.of(orderIds, Defect.class)
                .checkEach(inSet(currentYandexOrdersIds))
                .getResult();
    }

    ValidationResult<List<YandexAgencyOrder>, Defect> validateOrdersAreNotCompleted(
            List<YandexAgencyOrder> currentYandexOrders) {
        return ListValidationBuilder.of(currentYandexOrders, Defect.class)
                .checkEach(currentStatusWeakerThenPaid())
                .getResult();
    }

    private Constraint<YandexAgencyOrder, Defect> currentStatusWeakerThenPaid() {
        return fromPredicate(order -> {
            //noinspection RedundantIfStatement
            if (order.getYaOrderStatus().equals(Status.PAID) || order.getYaOrderStatus().equals(Status.COMPLETED)) {
                return false;
            }
            return true;
        }, CommonDefects.invalidValue());
    }
}
