package ru.yandex.direct.intapi.entity.balanceclient.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.promocodes.model.CampPromocodes;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeClientDomain;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeDomainsCheckResult;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeInfo;
import ru.yandex.direct.core.entity.promocodes.model.TearOffReason;
import ru.yandex.direct.core.entity.promocodes.repository.CampPromocodesRepository;
import ru.yandex.direct.core.entity.promocodes.service.PromocodesAntiFraudService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyPromocodeParameters;
import ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyPromocodeValidationService;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.lettuce.LettuceLockBuilder;
import ru.yandex.direct.utils.TimeProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.ANTIFRAUD_PROMOCODES;

/**
 * Сервис обработки нотификаций о промокодах.
 *
 * @see <a href="https://st.yandex-team.ru/BALANCE-28045">BALANCE-28045: Антифрод по промокодам Директа</a>
 */
@ParametersAreNonnullByDefault
@Service
public class NotifyPromocodeService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyPromocodeService.class);

    private static final Comparator<PromocodeInfo> PROMOCODES_COMPARATOR = Comparator
            .comparing(PromocodeInfo::getInvoiceEnabledAt)
            .thenComparing(PromocodeInfo::getInvoiceId)
            .thenComparing(PromocodeInfo::getId);

    private static final int CONCURRENT_ERROR_CODE = 3001;
    private static final int INVALID_PARAMS_ERROR_CODE = 3010;
    private static final String CONCURRENT_ERROR_MESSAGE =
            "Processing other notification for this order at this moment, retry later";
    private static final String FATAL_ERROR_MESSAGE = "Failed to process notification";

    public static final String NOTIFY_PROMOCODE_LOCK_BUILDER = "NotifyPromocodeLockBuilder";

    private final NotifyPromocodeValidationService validationService;
    private final PromocodesAntiFraudService antiFraudService;
    private final CampPromocodesRepository campPromocodesRepository;
    private final ShardHelper shardHelper;
    private final LockBuilder lockBuilder;
    private final ValidationResultConversionService validationResultConversionService;
    private final TimeProvider timeProvider;
    private final CampaignRepository campaignRepository;

    @Autowired
    public NotifyPromocodeService(NotifyPromocodeValidationService validationService,
                                  PromocodesAntiFraudService antiFraudService,
                                  CampPromocodesRepository campPromocodesRepository,
                                  CampaignRepository campaignRepository,
                                  ShardHelper shardHelper,
                                  @Qualifier(NOTIFY_PROMOCODE_LOCK_BUILDER) LockBuilder lockBuilder,
                                  ValidationResultConversionService validationResultConversionService) {
        this(validationService, antiFraudService, campPromocodesRepository, campaignRepository, shardHelper,
                lockBuilder, validationResultConversionService, new TimeProvider());
    }

    /**
     * Конструктор для тестов
     */
    @SuppressWarnings("checkstyle:parameternumber")
    NotifyPromocodeService(
            NotifyPromocodeValidationService validationService,
            PromocodesAntiFraudService antiFraudService,
            CampPromocodesRepository campPromocodesRepository,
            CampaignRepository campaignRepository,
            ShardHelper shardHelper,
            LockBuilder lockBuilder,
            ValidationResultConversionService validationResultConversionService,
            TimeProvider timeProvider
    ) {
        this.validationService = validationService;
        this.antiFraudService = antiFraudService;
        this.campPromocodesRepository = campPromocodesRepository;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.lockBuilder = lockBuilder;
        this.validationResultConversionService = validationResultConversionService;
        this.timeProvider = timeProvider;
    }

    /**
     * Обработчик нотификации.
     * Валидирует входные данные, берет блокировку на номер кампании в Redis'е и под ей выполняет работу.
     *
     * @param updateRequest балансовая нотификация
     * @return ответ балансу: успех или ошибка
     */
    public BalanceClientResponse notifyPromocode(NotifyPromocodeParameters updateRequest) {
        // Валидация входных данных
        ValidationResult<NotifyPromocodeParameters, Defect> validationResult =
                validationService.validateRequest(updateRequest);
        if (validationResult.hasAnyErrors()) {
            return validationErrorResponse(validationResult);
        }

        if (shouldIgnoreCampPromocode(updateRequest.getCampaignId())) {
            return BalanceClientResponse.success();
        }

        DistributedLock lock = lockBuilder.build(updateRequest);
        try {
            lock.tryLock();
        } catch (DistributedLockException e) {
            logger.error("cant acquire lock:", e);
        }
        if (!lock.isLocked()) {
            logger.warn("Failed to get lock for campaign {}", updateRequest.getCampaignId());
            return BalanceClientResponse.errorConflict(CONCURRENT_ERROR_CODE, CONCURRENT_ERROR_MESSAGE);
        }

        try {
            processNotification(updateRequest);
        } catch (Exception e) {
            logger.error(FATAL_ERROR_MESSAGE, e);
            return BalanceClientResponse.criticalError(FATAL_ERROR_MESSAGE);
        } finally {
            lock.unlock();
        }

        return BalanceClientResponse.success();
    }

    private Campaign getCampaign(long campaignId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, Collections.singletonList(campaignId));
        return Iterables.getOnlyElement(campaigns);
    }

    /**
     * Проверяет, не нужно ли проигнорировать нотификацию о промокоде.
     * Нужно, если это кампания под ОС, которой разрешены промокоды, или биллинговый агрегат.
     *
     * @return true, если нужно проигнорировать промокод.
     */
    boolean shouldIgnoreCampPromocode(long campaignId) {
        Campaign campaign = getCampaign(campaignId);

        CampaignType campType = campaign.getType();
        // игнорируем нотификацию на кампанию под общим счётом, т.к. уже приходила нотификация на сам ОС
        // обработка нотификаций на биллинговые агрегаты отрывает промокоды, т.к. у них нет доменов
        if (campaign.getWalletId() > 0 &&
                (ANTIFRAUD_PROMOCODES.contains(campType) || campType == CampaignType.BILLING_AGGREGATE)) {
            logger.info("Ignoring promocode for campaign {} because it is under wallet", campaignId);
            return true;
        }

        return false;
    }

    /**
     * Непосредственная работа с нотификацией.
     * Из списка промокодов отбрасываем нерелевантные, оставшиеся синхронизируем с БД либо отрываем промокод.
     *
     * @param updateRequest балансовая нотификация
     */
    void processNotification(NotifyPromocodeParameters updateRequest) {
        int shard = shardHelper.getShardByCampaignId(updateRequest.getCampaignId());

        CampPromocodes dbCampaignPromocodes =
                campPromocodesRepository.getCampaignPromocodes(shard, updateRequest.getCampaignId());
        var promocodeClientDomains = antiFraudService
                .getNormalizedPromocodeDomains(updateRequest.getPromocodes().stream()
                        .map(BalancePromocodeInfo::getCode)
                        .filter(s -> !StringUtils.isBlank(s))
                        .collect(Collectors.toList()));

        List<PromocodeInfo> balancePromocodes = updateRequest.getPromocodes().stream()
                .filter(bpi -> isApplicablePromocode(bpi, promocodeClientDomains))
                .map(NotifyPromocodeService::convertBalancePromocodeInfoToModel)
                .sorted(PROMOCODES_COMPARATOR)
                .collect(Collectors.toList());
        logger.trace("Applicable promocodes: {}", balancePromocodes);

        if (dbCampaignPromocodes == null) {
            if (balancePromocodes.isEmpty()) {
                // пришли какие-то промокоды, но ни один из них нас не касается
                return;
            }

            String restrictedDomain = antiFraudService.determineRestrictedDomain(updateRequest.getCampaignId());
            if (restrictedDomain == null) {
                // нет уникального домена для фиксации - отрываем промокоды
                antiFraudService.tearOffPromocodes(updateRequest.getServiceId(), updateRequest.getCampaignId(),
                        balancePromocodes, TearOffReason.BAD_DOMIAN);
            } else {
                List<PromocodeInfo> acceptableClientDomainPromocodes = processClientDomainPromocodes(balancePromocodes,
                        restrictedDomain,
                        updateRequest.getCampaignId(),
                        updateRequest.getServiceId(),
                        promocodeClientDomains);
                if (acceptableClientDomainPromocodes.isEmpty()) {
                    return;
                }

                var promocodesToSave = filterPromocodesToSave(acceptableClientDomainPromocodes,
                        updateRequest, restrictedDomain);
                if (promocodesToSave.isEmpty()) {
                    return;
                }

                // добавляем новую ограничивающую запись
                CampPromocodes newRecord = new CampPromocodes()
                        .withCampaignId(updateRequest.getCampaignId())
                        .withPromocodes(promocodesToSave)
                        .withRestrictedDomain(restrictedDomain)
                        .withLastChange(timeProvider.now());
                logger.info("Add restrictions for campaign {}: domain is {}", updateRequest.getCampaignId(),
                        restrictedDomain);
                campPromocodesRepository.addCampaignPromocodes(shard, newRecord);
            }
        } else {
            if (balancePromocodes.isEmpty()) {
                // удаляем ограничивающую запись, так как промокоды потрачены
                logger.info("Remove restrictions for campaign {}", updateRequest.getCampaignId());
                campPromocodesRepository.deleteCampaignPromocodes(shard, updateRequest.getCampaignId());
            } else if (!balancePromocodes.equals(dbCampaignPromocodes.getPromocodes())) {
                // обновляем список промокодов
                logger.info("Update promocodes list for campaign {}", updateRequest.getCampaignId());
                campPromocodesRepository
                        .updateCampaignPromocodesList(shard, updateRequest.getCampaignId(), balancePromocodes);
            }
        }
    }

    List<PromocodeInfo> filterPromocodesToSave(List<PromocodeInfo> promocodeInfos,
                                               NotifyPromocodeParameters updateRequest,
                                               String restrictedDomain) {
        if (antiFraudService.isAcceptableWallet(getCampaign(updateRequest.getCampaignId()))) {
            logger.info("Adding a domain under a recently created wallet");
        } else if (antiFraudService.domainHasStat(restrictedDomain)) {
            var promocodesToTearOff = promocodeInfos.stream()
                    .filter(p -> Boolean.TRUE.equals(p.getForNewClientsOnly()))
                    .collect(Collectors.toList());
            if (!promocodesToTearOff.isEmpty()) {
                logger.warn("Attempt to add domain {} with impressions as restricted", restrictedDomain);
                antiFraudService.tearOffPromocodes(updateRequest.getServiceId(), updateRequest.getCampaignId(),
                        promocodesToTearOff, TearOffReason.HAS_IMPRESSIONS);
                return promocodeInfos.stream()
                        .filter(p -> !Boolean.TRUE.equals(p.getForNewClientsOnly()))
                        .collect(Collectors.toList());
            }
        }
        return promocodeInfos;
    }

    /**
     * Нужно ли применять антифрод к данному промокоду. Условия:<ul>
     * <li>Промокод активирован после граничной даты, получаемой из {@link PromocodesAntiFraudService} </li>
     * <li>Промокод применим только к уникальным доменам или у нас есть запись с ним в ppcdict.promocode_domains</li>
     * <li>В прокомоде есть доступные незаакченные средства</li>
     * </ul>
     *
     * @param balancePromocodeInfo балансовые сведения о промокоде
     * @param promocodeClientDomains — словарь из нормализованных промокодов
     * @return {@code true} если нужно, {@code false} иначе
     */
    boolean isApplicablePromocode(BalancePromocodeInfo balancePromocodeInfo,
                                  Map<String, PromocodeClientDomain> promocodeClientDomains) {
        var promocode = balancePromocodeInfo.getCode();
        return !StringUtils.isBlank(promocode)
                && (balancePromocodeInfo.getUniqueUrlNeeded()
                    || promocodeClientDomains.containsKey(PromocodesAntiFraudService.normalizePromocode(promocode)))
                && balancePromocodeInfo.getInvoiceEnabledAt().isAfter(antiFraudService.getAntiFraudStartDate())
                && balancePromocodeInfo.getAvailableQty().compareTo(BigDecimal.ZERO) > 0;
    }

    /**
     * Выполнить проверку соответствия домена и клиента промокода с данными Комдепа.
     * При необходимости несовпавшие промокоды оторвать, вернуть те, что не отрывали.
     *
     * @param balancePromocodes — промокоды на проверку
     * @param domain — домен
     * @param campaignId — кампания, на которую зачисляются промокоды
     * @param serviceId — id сервиса в Балансе
     * @return — список промокодов, прошедших проверку без отрыва
     */
    private List<PromocodeInfo> processClientDomainPromocodes(List<PromocodeInfo> balancePromocodes,
                                                              String domain,
                                                              Long campaignId,
                                                              Integer serviceId,
                                                              Map<String, PromocodeClientDomain> promocodeDomains) {
        LinkedHashSet<PromocodeInfo> acceptableClientDomains = getAcceptableClientDomains(
                balancePromocodes,
                domain,
                campaignId,
                serviceId,
                promocodeDomains
        );
        List<PromocodeInfo> promocodesToTearOff =
                unacceptableClientDomainPromocodes(balancePromocodes, acceptableClientDomains);
        if (!promocodesToTearOff.isEmpty()) {
            antiFraudService.tearOffPromocodes(serviceId,
                    campaignId,
                    promocodesToTearOff,
                    TearOffReason.MISMATCH);
        }
        return new ArrayList<>(acceptableClientDomains);
    }

    private LinkedHashSet<PromocodeInfo> getAcceptableClientDomains(List<PromocodeInfo> balancePromocodes,
                                                            String domain,
                                                            Long campaignId, // наличие кампании уже проверено
                                                            Integer serviceId,
                                                            Map<String, PromocodeClientDomain> promocodeDomains) {
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(campaignId));
        ArrayList<String> codesToCheck = balancePromocodes.stream().map(PromocodeInfo::getCode)
                .collect(Collectors.toCollection(ArrayList::new)); // хотим быть уверены в мутабельности списка
        Map<String, PromocodeDomainsCheckResult> checkResultMap =
                antiFraudService.checkPromocodeDomains(codesToCheck, clientId, domain, promocodeDomains);
        LinkedHashSet<PromocodeInfo> answer = new LinkedHashSet<>();

        codesToCheck.removeAll(checkResultMap.keySet());
        if (!codesToCheck.isEmpty()) { // не должно происходить
            logger.error("Got no results for codes {} from checkPromocodeDomains!", codesToCheck);
        }

        boolean mismatchFound = false;
        for (PromocodeInfo balancePromocode : balancePromocodes) {
            PromocodeDomainsCheckResult promocodeDomainsCheckResult = checkResultMap.get(balancePromocode.getCode());
            if (promocodeDomainsCheckResult != null && promocodeDomainsCheckResult.isMismatch()) {
                mismatchFound = true;
            } else {
                answer.add(balancePromocode);
            }
        }

        if (mismatchFound && antiFraudService
                .shouldTearOffMismatched(serviceId,
                                         campaignId,
                                         unacceptableClientDomainPromocodes(balancePromocodes, answer))) {
            return answer;
        }

        return new LinkedHashSet<>(balancePromocodes);
    }

    private List<PromocodeInfo> unacceptableClientDomainPromocodes(List<PromocodeInfo> allCodes,
                                                                   Set<PromocodeInfo> acceptableCodes) {
        return allCodes.stream().filter(i -> !acceptableCodes.contains(i)).collect(Collectors.toList());
    }

    private static PromocodeInfo convertBalancePromocodeInfoToModel(BalancePromocodeInfo balancePromocodeInfo) {
        return new PromocodeInfo()
                .withId(balancePromocodeInfo.getId())
                .withCode(balancePromocodeInfo.getCode())
                .withForNewClientsOnly(balancePromocodeInfo.getForNewClientsOnly())
                .withInvoiceId(balancePromocodeInfo.getInvoiceId())
                .withInvoiceExternalId(balancePromocodeInfo.getInvoiceExternalId())
                .withInvoiceEnabledAt(balancePromocodeInfo.getInvoiceEnabledAt());
    }

    private static String formatError(IntapiError error) {
        return String.format("%s (%s)", error.getText(), error.getPath());
    }

    private BalanceClientResponse validationErrorResponse(
            ValidationResult<NotifyPromocodeParameters, Defect> validationResult) {
        String errors = validationResultConversionService.buildIntapiValidationResult(validationResult)
                .getErrors()
                .stream()
                .map(NotifyPromocodeService::formatError)
                .collect(joining("; "));
        return BalanceClientResponse.error(INVALID_PARAMS_ERROR_CODE, errors);
    }

    @Component(NOTIFY_PROMOCODE_LOCK_BUILDER)
    public static class LockBuilder {
        private static final Duration LOCK_TTL = Duration.ofMinutes(10);
        private static final String LOCK_PREFIX = "NotifyPromocode";

        private final LettuceLockBuilder lockBuilder;

        @Autowired
        public LockBuilder(@Qualifier(LETTUCE) LettuceConnectionProvider lettuce,
                           @Value("${redis.key_prefix}") String basePrefix) {
            lockBuilder = LettuceLockBuilder.newBuilder(lettuce::getConnection)
                    .withMaxLocks(1)
                    .withTTL(LOCK_TTL.toMillis())
                    .withKeyPrefix(basePrefix + LOCK_PREFIX);
        }

        /**
         * Создать лок соответствующий номеру кампании в запросе
         *
         * @param updateRequest параметры запроса
         * @return инстанс лока с временем жизни {@link #LOCK_TTL}
         */
        public DistributedLock build(NotifyPromocodeParameters updateRequest) {
            String lockName = String.format("%d-%d", updateRequest.getServiceId(), updateRequest.getCampaignId());

            return lockBuilder.createLock(lockName);
        }
    }
}
