package ru.yandex.direct.intapi.entity.balanceclient.service.validation;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters;

import static java.lang.String.format;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters.CASH_BACK_BONUS_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters.CASH_BACK_CONSUMED_BONUS_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters.CLIENT_ID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyClientCashBackParameters.CURRENCY_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.INVALID_SERVICE_ID_ERROR_CODE;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.INVALID_SERVICE_MESSAGE;
import static ru.yandex.direct.intapi.entity.balanceclient.service.validation.NotifyOrderValidationService.invalidFieldMessage;

@ParametersAreNonnullByDefault
@Service
public class NotifyClientCashBackValidationService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyClientCashBackValidationService.class);

    static final int INVALID_CASH_BACK_BONUS_ERROR_CODE = 1050;
    static final int INVALID_CLIENT_ID_ERROR_CODE = 1051;
    static final int INVALID_CURRENCY_ERROR_CODE = 1052;
    static final int INVALID_CONSUMED_CASH_BACK_BONUS_ERROR_CODE = 1053;


    private final Set<Integer> serviceIdsWithCurrencyValidation;

    @Autowired
    public NotifyClientCashBackValidationService(@Value("${balance.directServiceId}") int directServiceId) {
        serviceIdsWithCurrencyValidation = Set.of(directServiceId);
    }

    /**
     * Первичная проверка данных из запроса
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateRequest(NotifyClientCashBackParameters parameters) {

        Integer serviceId = parameters.getServiceId();
        if (serviceId == null || !serviceIdsWithCurrencyValidation.contains(serviceId)) {
            String message = format(INVALID_SERVICE_MESSAGE, serviceId);
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_SERVICE_ID_ERROR_CODE, message);
        }

        if (parameters.getClientId() == null || parameters.getClientId() < 0) {
            String message = invalidFieldMessage(CLIENT_ID_FIELD_NAME, parameters.getCashBackBonus());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CLIENT_ID_ERROR_CODE, message);
        }

        // баланс может прислать отрицательное значение кэшбека, считаем это допустимым
        if (parameters.getCashBackBonus() == null) {
            String message = invalidFieldMessage(CASH_BACK_BONUS_FIELD_NAME, parameters.getCashBackBonus());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CASH_BACK_BONUS_ERROR_CODE, message);
        }

        //баланс может прислать отрицательное значение кэшбека, считаем это допустимым
        if (parameters.getCashbackConsumedBonus() == null) {
            String message = invalidFieldMessage(CASH_BACK_CONSUMED_BONUS_FIELD_NAME,
                    parameters.getCashbackConsumedBonus());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CONSUMED_CASH_BACK_BONUS_ERROR_CODE, message);
        }

        String currency = parameters.getBalanceCurrency();
        if (currency == null || !Currencies.currencyExists(currency)) {
            String message = invalidFieldMessage(CURRENCY_FIELD_NAME, parameters.getCashBackBonus());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CURRENCY_ERROR_CODE, message);
        }

        return null;
    }
}
