package ru.yandex.direct.intapi.entity.balanceclient.service.validation;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.client.service.ClientCurrencyConversionTeaserService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.balanceclient.container.BalanceClientResponse;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters;

import static java.lang.String.format;
import static java.util.Map.entry;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.CPM;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.GEO;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.MEDIA;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.TEXT_CAMPAIGN_IN_BALANCE;
import static ru.yandex.direct.currency.CurrencyCode.QUASICURRENCY_CODES;
import static ru.yandex.direct.intapi.entity.balanceclient.model.BalanceClientResult.INTERNAL_ERROR_CODE;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.CAMPAIGN_ID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.CHIPS_COST_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.CHIPS_SPENT_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.SUM_REAL_MONEY_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.SUM_UNITS_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.TID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.TOTAL_SUM_FIELD_NAME;

@ParametersAreNonnullByDefault
@Service
public class NotifyOrderValidationService {
    private static final Logger logger = LoggerFactory.getLogger(NotifyOrderValidationService.class);

    static final String INVALID_FIELD_MESSAGE = "invalid %s from balance: %s, must be >= 0";
    static final String CAMPAIGN_IS_EMPTY_CANT_UPDATE_SUM_MESSAGE = "Campaign %d is empty, can't update sum: %s";
    static final String CAMPAIGN_IS_EMPTY_SKIPPING_ZERO_SUM_BUT_ACCEPTING_NOTIFICATION_MESSAGE =
            "Campaign %d is empty, skipping zero sum, but accepting notification";
    public static final String INVALID_SERVICE_MESSAGE = "invalid service %s";
    static final String NO_VALID_TID_MESSAGE = "No valid %s given";
    static final String INVALID_CAMPAIGN_ID_MESSAGE = "Invalid %s from balance: %s";
    static final String CLIENT_IS_GOING_TO_CURRENCY_CONVERT_SOON_WILL_ACCEPT_NOTIFICATIONS_AFTER_ITS_DONE_MESSAGE =
            "Client %d is going to currency convert soon, will accept notifications after it's done";
    static final String CAMPAIGN_TYPE_DOES_NOT_CORRESPOND_TO_SERVICE_MESSAGE =
            "Campaign %d with type %s does not correspond to service %d";
    static final String NO_PRODUCT_CURRENCY_GIVEN_FOR_DIRECT_CAMPAIGN_MESSAGE =
            "No ProductCurrency given for Direct campaign %d";
    static final String OUR_PRODUCT_CURRENCY_DOESNT_MATCH_BALANCE_PRODUCT_CURRENCY_FOR_CAMPAIGN_MESSAGE =
            "Our product currency %s doesn't match Balance product currency %s for campaign %d";

    public static final int INVALID_SERVICE_ID_ERROR_CODE = 1008;
    static final int INVALID_SUM_UNITS_ERROR_CODE = 1009;
    static final int INVALID_CHIPS_COST_ERROR_CODE = 1010;
    static final int INVALID_SUM_REAL_MONEY_ERROR_CODE = 1011;
    static final int INVALID_CHIPS_SPENT_ERROR_CODE = 1012;
    static final int INVALID_TID_ERROR_CODE = 1013;
    static final int INCOMPATIBLE_SERVICE_ID_AND_CAMPAIGN_TYPE_ERROR_CODE = 1014;
    static final int INVALID_TOTAL_SUM_ERROR_CODE = 1015;
    static final int INVALID_CAMPAIGN_ID_ERROR_CODE = 1016;
    static final int SUM_ON_EMPTY_CAMPAIGN_ID_ERROR_CODE = 4010;

    private final ClientCurrencyConversionTeaserService clientCurrencyConversionTeaserService;

    private final Set<Integer> serviceIdsWithCurrencyValidation;
    private final Map<Integer, Set<CampaignType>> serviceIdToCampaignTypes;

    public NotifyOrderValidationService(ClientCurrencyConversionTeaserService clientCurrencyConversionTeaserService,
                                        @Value("${balance.directServiceId}") int directServiceId,
                                        @Value("${balance.bayanServiceId}") int bayanServiceId,
                                        @Value("${balance.bananaServiceId}") int bananaServiceId) {
        this.clientCurrencyConversionTeaserService = clientCurrencyConversionTeaserService;

        serviceIdToCampaignTypes = Map.ofEntries(
                entry(directServiceId, ImmutableSet.<CampaignType>builder()
                        .addAll(GEO)
                        .addAll(TEXT_CAMPAIGN_IN_BALANCE)
                        .addAll(CPM)
                        .add(CampaignType.BILLING_AGGREGATE)
                        .build()),
                entry(bayanServiceId, MEDIA),
                entry(bananaServiceId, Set.of(CampaignType.INTERNAL_AUTOBUDGET)));

        serviceIdsWithCurrencyValidation = Set.of(directServiceId, bananaServiceId);
    }

    public static String invalidFieldMessage(String field, @Nullable BigDecimal value) {
        String valueRepr = value == null ? "undef" : value.toString();
        return format(INVALID_FIELD_MESSAGE, field, valueRepr);
    }

    /**
     * Проверка на номер транзакции (должен все время возрастать)
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public static BalanceClientResponse validateCampaignTid(NotifyOrderParameters parameters, Long campaignBalanceTid) {
        if (campaignBalanceTid > parameters.getTid()) {
            logger.warn("Can't apply ('{}', {}, {}, {}), current tid: {}",
                    parameters.getServiceId(),
                    parameters.getCampaignId(),
                    parameters.getSumUnits(),
                    parameters.getTid(),
                    campaignBalanceTid);
            return BalanceClientResponse.success();
        }
        return null;
    }

    /**
     * Проверка, что на пустую (недосозданную/удаленную) кампанию не пытаются зачислить денег
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateSumOnEmptyCampaign(boolean campaignStatusEmpty, Money sum,
                                                            Long campaignId) {
        if (campaignStatusEmpty) {
            if (sum.greaterThanZero()) {
                String message = format(CAMPAIGN_IS_EMPTY_CANT_UPDATE_SUM_MESSAGE, campaignId, sum.bigDecimalValue());
                logger.warn(message);
                return BalanceClientResponse.error(SUM_ON_EMPTY_CAMPAIGN_ID_ERROR_CODE, message);
            } else {
                String message =
                        format(CAMPAIGN_IS_EMPTY_SKIPPING_ZERO_SUM_BUT_ACCEPTING_NOTIFICATION_MESSAGE, campaignId);
                logger.warn(message);
                return BalanceClientResponse.success(message);
            }
        }
        return null;
    }

    /**
     * Проверка что TotalConsumeQty валидна
     */
    public static BalanceClientResponse validateTotalSum(@Nullable BigDecimal totalSum) {
        if (totalSum == null || totalSum.compareTo(BigDecimal.ZERO) < 0) {
            String message = invalidFieldMessage(TOTAL_SUM_FIELD_NAME, totalSum);
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_TOTAL_SUM_ERROR_CODE, message);
        }

        return null;
    }

    /**
     * Первичная проверка данных из запроса
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateRequest(NotifyOrderParameters parameters) {
        Integer serviceId = parameters.getServiceId();
        if (serviceId == null || !serviceIdToCampaignTypes.containsKey(serviceId)) {
            String message = format(INVALID_SERVICE_MESSAGE, serviceId);
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_SERVICE_ID_ERROR_CODE, message);
        }

        if (parameters.getSumUnits() == null || parameters.getSumUnits().compareTo(BigDecimal.ZERO) < 0) {
            String message = invalidFieldMessage(SUM_UNITS_FIELD_NAME, parameters.getSumUnits());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_SUM_UNITS_ERROR_CODE, message);
        }

        if (parameters.getChipsCost() != null && parameters.getChipsCost().compareTo(BigDecimal.ZERO) < 0) {
            String message = invalidFieldMessage(CHIPS_COST_FIELD_NAME, parameters.getChipsCost());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CHIPS_COST_ERROR_CODE, message);
        }

        if (parameters.getSumRealMoney() != null && parameters.getSumRealMoney().compareTo(BigDecimal.ZERO) < 0) {
            String message = invalidFieldMessage(SUM_REAL_MONEY_FIELD_NAME, parameters.getSumRealMoney());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_SUM_REAL_MONEY_ERROR_CODE, message);
        }

        if (parameters.getChipsSpent() != null && parameters.getChipsSpent().compareTo(BigDecimal.ZERO) < 0) {
            String message = invalidFieldMessage(CHIPS_SPENT_FIELD_NAME, parameters.getChipsSpent());
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_CHIPS_SPENT_ERROR_CODE, message);
        }

        if (parameters.getTid() == null) {
            String message = format(NO_VALID_TID_MESSAGE, TID_FIELD_NAME);
            logger.warn(message);
            return BalanceClientResponse.error(INVALID_TID_ERROR_CODE, message);
        }

        if (parameters.getCampaignId() == null || parameters.getCampaignId() <= 0) {
            return BalanceClientResponse.error(INVALID_CAMPAIGN_ID_ERROR_CODE,
                    format(INVALID_CAMPAIGN_ID_MESSAGE, CAMPAIGN_ID_FIELD_NAME, parameters.getCampaignId()));
        }

        return null;
    }

    /**
     * Перестаём принимать нотификации на заказы за N минут до начала конвертации
     * и начинаем принимать их снова только после её окончания
     * иначе можем записать сумму не в валюте кампании
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateClientCurrencyConversionState(Long clientId) {
        if (clientCurrencyConversionTeaserService.isClientConvertingSoon(ClientId.fromLong(clientId))) {
            String message =
                    format(CLIENT_IS_GOING_TO_CURRENCY_CONVERT_SOON_WILL_ACCEPT_NOTIFICATIONS_AFTER_ITS_DONE_MESSAGE,
                            clientId);
            logger.warn(message);
            return BalanceClientResponse.errorLocked(INTERNAL_ERROR_CODE, message);
        }

        return null;
    }

    /**
     * Проверка на соответствие ServiceID типу кампании
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateCampaignType(Integer serviceId, CampaignType campaignType, Long campaignId) {
        Set<CampaignType> expectedKind = serviceIdToCampaignTypes.get(serviceId);

        if (expectedKind == null || !expectedKind.contains(campaignType)) {
            String message = format(CAMPAIGN_TYPE_DOES_NOT_CORRESPOND_TO_SERVICE_MESSAGE,
                    campaignId, campaignType.name().toLowerCase(), serviceId);
            logger.warn(message);
            return BalanceClientResponse.error(INCOMPATIBLE_SERVICE_ID_AND_CAMPAIGN_TYPE_ERROR_CODE, message);
        }

        return null;
    }

    /**
     * Проверка соответствия валюты продукта
     *
     * @return {@code null} для нотификации, которую можно продолжать обрабатывать, иначе - ответ балансу
     */
    public BalanceClientResponse validateProductCurrency(int serviceId, @Nullable String productCurrency,
                                                         CurrencyCode dbProductCurrency, Long campaignId) {
        if (!serviceIdsWithCurrencyValidation.contains(serviceId)) {
            return null;
        }

        if (productCurrency == null) {
            String message = format(NO_PRODUCT_CURRENCY_GIVEN_FOR_DIRECT_CAMPAIGN_MESSAGE, campaignId);
            logger.warn(message);
            return BalanceClientResponse.criticalError(message);
        } else if (dbProductCurrency != null && QUASICURRENCY_CODES.contains(dbProductCurrency)
                && productCurrency.isEmpty()) {
            /*
                https://st.yandex-team.ru/BALANCE-23760,
                https://st.yandex-team.ru/DIRECT-73517.
                Легальный случай, принимаем такие нотификации
             */
            return null;
        } else {
            CurrencyCode currencyForCompare;
            if (productCurrency.isEmpty()) {
                currencyForCompare = CurrencyCode.YND_FIXED;
            } else if (Currencies.currencyExists(productCurrency)) {
                currencyForCompare = CurrencyCode.valueOf(productCurrency);
            } else {
                currencyForCompare = null;
            }

            if (!dbProductCurrency.equals(currencyForCompare)) {
                String message = format(
                        OUR_PRODUCT_CURRENCY_DOESNT_MATCH_BALANCE_PRODUCT_CURRENCY_FOR_CAMPAIGN_MESSAGE,
                        dbProductCurrency, productCurrency, campaignId);
                logger.warn(message);
                return BalanceClientResponse.criticalError(message);
            }
        }

        return null;
    }
}
