package ru.yandex.direct.intapi.entity.balanceclient.service.validation;

import java.math.BigDecimal;

import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo;
import ru.yandex.direct.intapi.entity.balanceclient.model.NotifyPromocodeParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.NumberConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo.AVAILABLE_PROMOCODE_QTY_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo.NEED_UNIQUE_URLS_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo.PROMOCODE_ID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.BalancePromocodeInfo.START_DATE_TIME_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.CAMPAIGN_ID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyOrderParameters.SERVICE_ID_FIELD_NAME;
import static ru.yandex.direct.intapi.entity.balanceclient.model.NotifyPromocodeParameters.PROMOCODES_FIELD_NAME;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
public class NotifyPromocodeValidationService {
    private static final Constraint<BigDecimal, Defect> NOT_LESS_THAN_ZERO =
            NumberConstraints.notLessThan(BigDecimal.ZERO);

    private final Constraint<Integer, Defect> serviceIdIsAllowed;
    private final Constraint<Long, Defect> campaignExists;

    @Autowired
    public NotifyPromocodeValidationService(@Value("${balance.directServiceId}") int directServiceId,
                                            CampaignService campaignService) {
        ImmutableSet<Integer> allowedServiceIds = ImmutableSet.of(directServiceId);

        serviceIdIsAllowed = Constraint.fromPredicate(allowedServiceIds::contains, invalidValue());

        campaignExists = Constraint.fromPredicate(campaignService::campaignExists, campaignNotFound());
    }

    /**
     * Первичная проверка данных из запроса
     *
     * @return результат валидации
     * @implNote делает запрос в БД при проверке существования кампании
     */
    public ValidationResult<NotifyPromocodeParameters, Defect> validateRequest(NotifyPromocodeParameters parameters) {

        ItemValidationBuilder<NotifyPromocodeParameters, Defect> vb = ItemValidationBuilder.of(parameters);

        vb.item(parameters.getServiceId(), SERVICE_ID_FIELD_NAME)
                .check(notNull())
                .check(serviceIdIsAllowed);

        vb.item(parameters.getCampaignId(), CAMPAIGN_ID_FIELD_NAME)
                .check(notNull())
                .check(validId())
                .check(campaignExists, When.isValid());

        vb.list(parameters.getPromocodes(), PROMOCODES_FIELD_NAME)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(this::validatePromocodeInfo, When.notNull());

        return vb.getResult();
    }

    ValidationResult<BalancePromocodeInfo, Defect> validatePromocodeInfo(BalancePromocodeInfo promocodeInfo) {
        ItemValidationBuilder<BalancePromocodeInfo, Defect> vb = ItemValidationBuilder.of(promocodeInfo);

        vb.item(promocodeInfo.getId(), PROMOCODE_ID_FIELD_NAME)
                .check(notNull())
                .check(validId(), When.notNull());

        vb.item(promocodeInfo.getUniqueUrlNeeded(), NEED_UNIQUE_URLS_FIELD_NAME)
                .check(notNull());

        vb.item(promocodeInfo.getInvoiceEnabledAt(), START_DATE_TIME_FIELD_NAME)
                .check(notNull());

        vb.item(promocodeInfo.getAvailableQty(), AVAILABLE_PROMOCODE_QTY_FIELD_NAME)
                .check(notNull())
                .check(NOT_LESS_THAN_ZERO);

        return vb.getResult();
    }
}
