package ru.yandex.direct.intapi.entity.bidmodifiers.controller;

import java.util.List;

import com.google.common.collect.Multimap;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.common.converter.CampaignTypeConverter;
import ru.yandex.direct.intapi.validation.ErrorStatusOnValidationFail;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiSuccessResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.bidmodifiers.ComplexBidModifierConverter;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifiersListWebResponse;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.ComplexBidModifierWeb;
import ru.yandex.direct.web.core.model.WebResponse;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.intapi.entity.bidmodifiers.model.BidModifierResponse.buildBidModifierResponse;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_PROD;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_TEST;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToTopLevel;
import static ru.yandex.direct.web.core.entity.bidmodifiers.ComplexBidModifierConverter.convertToExternal;

@RestController
@RequestMapping("bidmodifiers")
@Api(tags = "intapiBidModifiers")
@AllowServices(production = {DIRECT_SCRIPTS_PROD, DIRECT_INTAPI_PROD, DIRECT_WEB_PROD, DIRECT_API_PROD, SURVEYS_PROD},
        testing = {DIRECT_SCRIPTS_TEST, DIRECT_INTAPI_TEST, DIRECT_WEB_TEST, DIRECT_API_TEST, DIRECT_AUTOTESTS,
                DIRECT_DEVELOPER, SURVEYS_TEST},
        sandbox = {DIRECT_SCRIPTS_SANDBOX, DIRECT_INTAPI_SANDBOX, DIRECT_API_SANDBOX},
        sandboxTesting = {DIRECT_SCRIPTS_SANDBOX_TEST, DIRECT_INTAPI_SANDBOX_TEST, DIRECT_API_SANDBOX_TEST,
                DIRECT_AUTOTESTS, DIRECT_DEVELOPER})
public class BidModifiersController {
    private final BidModifierService bidModifierService;
    private final ComplexBidModifierService complexBidModifierService;
    private final ValidationResultConversionService validationResultConversionService;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;

    private final CampaignTypeConverter campaignTypeConverter = new CampaignTypeConverter();

    @Autowired
    public BidModifiersController(BidModifierService bidModifierService,
                                  ComplexBidModifierService complexBidModifierService,
                                  ValidationResultConversionService validationResultConversionService,
                                  AdGroupRepository adGroupRepository,
                                  ShardHelper shardHelper) {
        this.bidModifierService = bidModifierService;
        this.complexBidModifierService = complexBidModifierService;
        this.validationResultConversionService = validationResultConversionService;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
    }

    @InitBinder
    public void initBinder(WebDataBinder dataBinder) {
        dataBinder.registerCustomEditor(CampaignType.class, campaignTypeConverter);
    }

    /**
     * Возвращает набор корректировок для выбранной кампании/группы.
     * Предназначено для вызова из перлового контроллера при отрисовке формы редактирования кампании.
     * (получение для группы поддерживается, но скорее всего пока не будет использоваться).
     */
    @ApiOperation(
            value = "intapiGetBidModifiers",
            httpMethod = "GET",
            nickname = "intapiGetBidModifiers"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = BidModifiersListWebResponse.class)
    })
    @RequestMapping(path = "/get", method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public BidModifiersListWebResponse getBidModifiers(
            @RequestParam("campaignId") long campaignId,
            @RequestParam(value = "adGroupId", required = false) Long adGroupId
    ) {
        List<BidModifier> bidModifiers;
        long clientId = shardHelper.getClientIdByCampaignId(campaignId);
        long operatorUid = shardHelper.getUidsByClientId(clientId).get(0);

        if (adGroupId != null) {
            bidModifiers =
                    bidModifierService.getByAdGroupIds(ClientId.fromLong(clientId), singleton(adGroupId), emptySet(),
                            Constants.ALL_TYPES, singleton(BidModifierLevel.ADGROUP), operatorUid);
        } else {
            bidModifiers = bidModifierService.getByCampaignIds(
                    ClientId.fromLong(clientId), singleton(campaignId),
                    Constants.ALL_TYPES, singleton(BidModifierLevel.CAMPAIGN), operatorUid);
        }
        return convertToExternal(bidModifiers);
    }

    /**
     * Выполняет валидацию набора корректировок для кампании (для группы не поддерживается).
     * Предназначено для вызова из перлового контроллера при сохранении кампании.
     * Возвращает результат валидации.
     */
    @ApiOperation(
            value = "intapiValidateBidModifiers",
            httpMethod = "POST",
            nickname = "intapiValidateBidModifiers"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiValidationResponse.class)
    })
    @RequestMapping(path = "/validate", method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ErrorStatusOnValidationFail
    @ResponseBody
    public IntapiValidationResponse validateBidModifiers(
            @RequestBody ComplexBidModifierWeb complexBidModifierWeb,
            @RequestParam("campaignType") CampaignType campaignType,
            @RequestParam("clientId") long clientId) {
        ValidationResult<ComplexBidModifierWeb, Defect> result =
                new ValidationResult<>(complexBidModifierWeb);

        ComplexBidModifier complexBidModifier = ComplexBidModifierConverter.fromWeb(complexBidModifierWeb);

        Pair<List<BidModifier>, Multimap<Integer, Integer>> flatModifiersInfo =
                complexBidModifierService.convertFromComplexModelsForCampaigns(singletonList(complexBidModifier));

        ValidationResult<List<BidModifier>, Defect> flatValidationResult =
                complexBidModifierService.validateBidModifiersFlat(
                        flatModifiersInfo.getLeft(), flatModifiersInfo.getRight(),
                        flatIndex -> campaignType,
                        // Так как ручка не поддерживает параметра adGroupId, то и на группы сохранять пока нельзя
                        flatIndex -> null,
                        ClientId.fromLong(clientId));

        ValidationResult<List<ComplexBidModifier>, Defect> validationResult =
                new ValidationResult<>(singletonList(complexBidModifier));
        complexBidModifierService.transferValidationResultFlatToComplex(
                validationResult, flatValidationResult, flatModifiersInfo.getRight());

        transferIssuesFromValidationToTopLevel(validationResult, result);

        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        return new IntapiValidationResponse(new IntapiValidationResult());
    }

    /**
     * Выполняет сохранение указанного набора корректировок на выбранную кампанию/группу.
     * Предназначено для вызова из перловых контроллеров при сохранении кампаний/групп.
     * Возвращает IntapiSuccessResponse если всё ок, объект с ошибками валидации в противном случае.
     */
    @ApiOperation(
            value = "intapiUpdateBidModifiers",
            httpMethod = "POST",
            nickname = "intapiUpdateBidModifiers"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })
    @RequestMapping(path = "/update", method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse updateBidModifiers(
            @RequestBody ComplexBidModifierWeb complexBidModifierWeb,
            @RequestParam("campaignId") long campaignId,
            @RequestParam(value = "adGroupId", required = false) Long adGroupId,
            @RequestParam("campaignType") CampaignType campaignType) {

        long clientId = shardHelper.getClientIdByCampaignId(campaignId);
        long operatorUid = shardHelper.getUidsByClientId(clientId).get(0);

        // Валидируем перед сохранением
        ValidationResult<ComplexBidModifierWeb, Defect> result = new ValidationResult<>(complexBidModifierWeb);

        ComplexBidModifier complexBidModifier = ComplexBidModifierConverter.fromWeb(complexBidModifierWeb);

        Pair<List<BidModifier>, Multimap<Integer, Integer>> flatModifiersInfo;
        AdGroup adGroupWithType;
        if (adGroupId != null) {
            flatModifiersInfo =
                    complexBidModifierService.convertFromComplexModelsForAdGroups(singletonList(complexBidModifier));
            int shard = shardHelper.getShardByCampaignId(campaignId);
            List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, singleton(adGroupId));
            adGroupWithType = adGroups.get(0);
        } else {
            flatModifiersInfo =
                    complexBidModifierService.convertFromComplexModelsForCampaigns(singletonList(complexBidModifier));
            adGroupWithType = null;
        }

        List<BidModifier> bidModifiers = flatModifiersInfo.getLeft();

        ValidationResult<List<BidModifier>, Defect> flatValidationResult =
                complexBidModifierService.validateBidModifiersFlat(
                        bidModifiers, flatModifiersInfo.getRight(), flatIndex -> campaignType,
                        flatIndex -> adGroupWithType,
                        ClientId.fromLong(clientId));

        ValidationResult<List<ComplexBidModifier>, Defect> validationResult =
                new ValidationResult<>(singletonList(complexBidModifier));
        complexBidModifierService.transferValidationResultFlatToComplex(
                validationResult, flatValidationResult, flatModifiersInfo.getRight());

        transferIssuesFromValidationToTopLevel(validationResult, result);

        // Если всё же есть ошибки в валидации -- не сохраняем, возвращаем validationResult
        if (result.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(result);
        }

        // Проставляем campaignId и adGroupId везде
        bidModifiers.forEach(it -> {
            it.setCampaignId(campaignId);
            it.setAdGroupId(adGroupId);
        });

        // Сохраняем изменения
        var affectedObjects = bidModifierService.replaceModifiers(ClientId.fromLong(clientId),
                operatorUid, bidModifiers,
                singleton(new CampaignIdAndAdGroupIdPair()
                        .withCampaignId(campaignId)
                        .withAdGroupId(adGroupId)));

        return buildBidModifierResponse(affectedObjects.size());
    }
}
