package ru.yandex.direct.intapi.entity.brandlift.service

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.HttpStatus
import org.springframework.stereotype.Service
import ru.yandex.direct.bannersystem.BannerSystemClient
import ru.yandex.direct.bannersystem.BsUriFactory
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportDimension
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportGroupByDate
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportMetric
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRequest
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRow
import ru.yandex.direct.core.entity.brandlift.repository.BrandSurveyRepository
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.intapi.ErrorResponse
import ru.yandex.direct.intapi.ErrorResponse.ErrorCode.NOT_FOUND
import ru.yandex.direct.intapi.IntApiException
import ru.yandex.direct.intapi.entity.brandlift.model.BrandliftStat
import ru.yandex.direct.utils.CommonUtils
import java.time.Duration
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import java.util.UUID
import java.util.stream.Collectors

@Service
class BrandliftService {
    @Autowired
    private lateinit var brandSurveyRepository : BrandSurveyRepository
    @Autowired
    private lateinit var shardHelper : ShardHelper
    @Autowired
    private lateinit var campaignRepository: CampaignRepository
    @Autowired
    private lateinit var bannerSystemClient: BannerSystemClient
    @Autowired
    private lateinit var clientService: ClientService

    companion object {
        private val TIMEOUT = Duration.ofSeconds(30)
        private val MASTER_REPORT_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd")
    }

    fun getStats(surveyId: String, clientIdLong: Long): BrandliftStat {
        val clientId = ClientId.fromLong(clientIdLong)
        val shard = shardHelper.getShardByClientId(clientId)
        if (shard == ShardSupport.NO_SHARD) {
            throw IntApiException(HttpStatus.NOT_FOUND, ErrorResponse(NOT_FOUND, "Not found client $clientId"))
        }
        var brandSurveyList = brandSurveyRepository.getBrandSurvey(shard, surveyId)
        if (brandSurveyList.isNullOrEmpty()) {
            // Если не находим brandLift на шарде клиента - может быть такое,
            // что brandLift скрытый и находится на шарде оператора, поэтому приходится идти по всем шардам
            brandSurveyList = shardHelper.dbShards().stream()
                .map { brandSurveyRepository.getBrandSurvey(it, surveyId) }
                .flatMap { it.stream() }
                .collect(Collectors.toList())
            if (brandSurveyList.isNullOrEmpty()) {
                throw IntApiException(HttpStatus.NOT_FOUND, ErrorResponse(NOT_FOUND, "Not found brandSurvey $surveyId"))
            }
        }
        val brandSurvey = brandSurveyList[0]
        val camps = campaignRepository.getCampaignsForBrandSurveys(shard, clientId, listOf(surveyId))[surveyId]
        if (camps.isNullOrEmpty()) {
            throw IntApiException(HttpStatus.NOT_FOUND, ErrorResponse(NOT_FOUND, "Not found campaigns"))
        }
        val startTime = camps.map { getCampaignDate(it, true) }.minOrNull()
        val finishTime = camps.map {  getCampaignDate(it, false) }.maxOrNull()
        val totalBsStats = getBsStats(clientId, startTime, finishTime, camps.map { it.orderId.toString() })
        return BrandliftStat(
            surveyId = surveyId,
            name = brandSurvey.name,
            shows = totalBsStats?.shows ?: 0L,// показов
            reach = totalBsStats?.uniqViewers ?: 0L,// охват
            startDate = DateTimeFormatter.ISO_DATE.format(startTime),
            endDate = DateTimeFormatter.ISO_DATE.format(finishTime),
            rf = totalBsStats?.rf,
            ctr = totalBsStats?.ctr,
            cpm = totalBsStats?.cpm?.div(1000000),//Ответ от БК в миллионных долях валюты
        )
    }

    private fun getBsStats(
        clientId: ClientId,
        from: LocalDate?,
        to: LocalDate?,
        orderIds: List<String>,
    ) : MasterReportRow? {
        val currency = clientService.getWorkCurrency(clientId).code.name
        val request = MasterReportRequest()
            .setWithVat(0)
            .setWithPerformanceCoverage(0)
            .setWithDiscount(1)
            .setLang("ru")
            .setGroupByDate(MasterReportGroupByDate.NONE)
            .setDontGroupAndFilterZerosForTotals(1)
            .setCountableFieldsByTargettype(emptyList())
            .setCurrency(currency)
            .setCountableFields(setOf(MasterReportMetric.SHOWS, MasterReportMetric.UNIQ_VIEWERS,
                MasterReportMetric.CTR, MasterReportMetric.CPM, MasterReportMetric.RF))
            .setGroupBy(setOf(MasterReportDimension.PAGE_NAME))
            .setDateFrom(MASTER_REPORT_DATE_FORMATTER.format(from))
            .setDateTo(MASTER_REPORT_DATE_FORMATTER.format(to))
            .setOrderIds(orderIds)
            .setLimits(
                MasterReportRequest.Limits().setLimit(10 + 1).setOffset(0)
            )
        val response = bannerSystemClient
            .doRequest(BsUriFactory.MASTER_REPORT, request, UUID.randomUUID(), TIMEOUT)
        return response?.getTotals()
    }

    private fun getCampaignDate(campaign: Campaign, start: Boolean): LocalDate {
        var date = if (start) campaign.startTime else campaign.finishTime
        val strategy = campaign.strategy
        if (strategy != null && strategy.strategyData != null) {
            val strategyName = strategy.strategyName
            val strategyData = strategy.strategyData
            if ((strategyName == StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD || strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD || strategyName == StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD) &&
                (start || date == null || CommonUtils.nvl(strategyData.autoProlongation, 0L) == 0L)
            ) {
                date = if (start) strategyData.start else strategyData.finish
            }
        }
        return date ?: LocalDate.now()
    }
}
