package ru.yandex.direct.intapi.entity.bs.export;

import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.bs.export.model.AssetHashes;
import ru.yandex.direct.core.entity.bs.export.queue.service.FullExportQueueService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.bs.export.service.BsExportGetAssetHashesService;
import ru.yandex.direct.intapi.entity.bs.export.service.BsExportLogsService;
import ru.yandex.direct.tvm.AllowServices;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_HANDLES;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.YABS_BANANA_PROD;
import static ru.yandex.direct.tvm.TvmService.YABS_BANANA_TEST;

@RestController
@RequestMapping("bsexport")
@Api(value = "Экспорт в БК")
@ParametersAreNonnullByDefault
public class BsExportController {

    private static final Logger logger = LoggerFactory.getLogger(BsExportController.class);

    private static final int MAX_ORDERIDS_COUNT = 1000;
    private static final int MIN_ORDERIDS_COUNT = 1;

    private final FullExportQueueService fullExportQueueService;
    private final ShardHelper shardHelper;
    private final BsExportGetAssetHashesService bsExportGetAssetHashesService;
    private final BsExportLogsService bsExportLogsService;


    @Autowired

    public BsExportController(FullExportQueueService fullExportQueueService, ShardHelper shardHelper,
                              BsExportGetAssetHashesService bsExportGetAssetHashesService,
                              BsExportLogsService bsExportLogsService) {
        this.fullExportQueueService = fullExportQueueService;
        this.shardHelper = shardHelper;
        this.bsExportGetAssetHashesService = bsExportGetAssetHashesService;
        this.bsExportLogsService = bsExportLogsService;
    }

    @AllowServices(testing = {DIRECT_AUTOTESTS}, production = {})
    @ApiOperation(value = "Добавить кампании в очередь ре-экспорта в контент-систему БК",
            httpMethod = "POST")
    @ApiResponses(
            {
                    @ApiResponse(code = 200, message = "Все найденные в БД кампании были успешно добавлены"),
                    @ApiResponse(code = 400, message = "Ни одна из переданных кампаний не была найдена/обработана",
                            response = ErrorResponse.class),
            }
    )
    @PostMapping(path = "add-campaigns-to-full-export-queue", consumes = APPLICATION_JSON)
    public void addToFullExportQueue(
            @ApiParam(value = "campaignIds", required = true)
            @RequestBody List<Long> campaignIds) {
        int result = fullExportQueueService.addCampaignsToFullExportQueue(campaignIds);
        if (!campaignIds.isEmpty() && result == 0) {
            ErrorResponse error = new ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "no campaigns found");
            throw new IntApiException(HttpStatus.BAD_REQUEST, error);
        }
    }

    @AllowServices(testing = {YABS_BANANA_TEST, DIRECT_DEVELOPER}, production = {YABS_BANANA_PROD})
    @ApiOperation(value = "Получить список сгенерированных OrderID",
            httpMethod = "POST", responseContainer = "List")
    @ApiResponses({
            @ApiResponse(code = 200, message = "список сгенерированных OrderID", response = List.class),
            @ApiResponse(code = 400, message = "ошибка запроса", response = ErrorResponse.class)})
    @PostMapping(path = "generate-order-ids", produces = APPLICATION_JSON)
    public List<Long> generateOrderIds(
            @ApiParam(value = "count", required = true)
            @RequestParam(value = "count") int count) {
        if (count < MIN_ORDERIDS_COUNT || count > MAX_ORDERIDS_COUNT) {
            ErrorResponse error =
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM,
                            "count param must be between " + MIN_ORDERIDS_COUNT + " and " + MAX_ORDERIDS_COUNT);
            throw new IntApiException(HttpStatus.BAD_REQUEST, error);
        }
        return shardHelper.generateBsOrderIds(count);
    }

    @AllowServices(testing = {DIRECT_HANDLES, DIRECT_AUTOTESTS, DIRECT_DEVELOPER}, production = {})
    @GetMapping(path = "legacyLogs", produces = APPLICATION_JSON)
    public Map<String, Map> getTestLogs(@RequestParam(value = "uuid") List<String> uuids,
                                        @RequestParam(value = "omitMdsMeta") boolean omitMdsMeta) {
        return bsExportLogsService.getLogs(uuids, omitMdsMeta);
    }

    @AllowServices(
            testing = {DIRECT_SCRIPTS_TEST, DIRECT_AUTOTESTS, DIRECT_DEVELOPER},
            production = {DIRECT_SCRIPTS_PROD}
    )
    @ApiOperation(
            value = "Получить хеши ассетов по переданным id кампаниям, сгруппированных по id баннерам",
            httpMethod = "POST",
            responseContainer = "Map"
    )
    @PostMapping(
            path = "get-campaigns-asset-hashes",
            consumes = APPLICATION_JSON,
            produces = APPLICATION_JSON
    )
    public Map<Long, Map<Long, AssetHashes>> getCampaignContents(
            @ApiParam(value = "campaignIds", required = true)
            @RequestBody List<Long> campaignIds) {
        return bsExportGetAssetHashesService.getCampaignsAssetHashes(new HashSet<>(campaignIds));
    }
}
