package ru.yandex.direct.intapi.entity.bs.export.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.grid.schema.yt.tables.TestBsexportTestLogs;
import ru.yandex.direct.intapi.entity.bs.export.model.TypeAndLogData;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static ru.yandex.direct.grid.schema.yt.Tables.TEST_BSEXPORT_TEST_LOGS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.toEntryStream;

/**
 * Репозиторий для работы с таблицей //home/direct/logs/bsexport, которая содержит логи отправки данных в БК
 * на непродакшен средах
 */
@Repository
public class BsExportLogsRepository {

    private static final TestBsexportTestLogs LOGS = TEST_BSEXPORT_TEST_LOGS.as("LOGS");
    private static final Field<String> UUID = aliased(LOGS.UUID);
    private static final Field<String> TYPE = aliased(LOGS.TYPE);
    private static final Field<String> DATA = aliased(LOGS.DATA);

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;

    @Autowired
    public BsExportLogsRepository(YtProvider ytProvider, DirectConfig directConfig) {
        this.ytProvider = ytProvider;
        DirectConfig config = directConfig.getBranch("bsexport_logs");
        this.ytCluster = YtCluster.valueOf(config.getString("cluster"));
    }

    /**
     * Получить логи отправок с uuid в {@param uuids}
     */
    public Map<String, List<TypeAndLogData>> getLogs(Collection<String> uuids) {
        var query = YtDSL.ytContext()
                .select(UUID, TYPE, DATA)
                .from(LOGS)
                .where(UUID.in(uuids));

        UnversionedRowset rowset = ytProvider.getDynamicOperator(ytCluster).selectRows(query);
        TableSchema schema = rowset.getSchema();

        Function<UnversionedRow, TypeAndLogData> valueMapper = (row) -> {
            String type = stringValueGetter(schema, TYPE).apply(row);
            String data = stringValueGetter(schema, DATA).apply(row);
            return new TypeAndLogData(type, data);
        };

        return toEntryStream(rowset, UUID, valueMapper)
                .mapKeys(UnversionedValue::stringValue)
                .grouping();
    }
}
