package ru.yandex.direct.intapi.entity.bs.export.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.bs.export.model.AssetHashes
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService

/**
 * Сервис для получения хешей ассетов для отправки в БК
 */
@Service
class BsExportGetAssetHashesService(
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val grutUacContentService: GrutUacContentService,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(BsExportGetAssetHashesService::class.java)
        private const val CHUNK_CAMPAIGNS_SIZE = 200
        private const val CHUNK_ADS_SIZE = 990L

        // Мы достигнем лимита попыток если во всех кампаниях будет >= 4950 ассетов
        // Если такое произойдет то можно уменьшить количество отправляемых кампаний
        // через проперти [CAMPAIGNS_CHUNK_FOR_GETTING_ASSET_HASHES]
        private const val MAX_COUNT_OF_REQUESTS = 1_000
    }

    fun getCampaignsAssetHashes(
        campaignIds: Set<Long>,
    ): Map<Long, Map<Long, AssetHashes>> {

        val assetHashes = campaignIds
            .chunked(CHUNK_CAMPAIGNS_SIZE)
            .map { getChunkOfCampaignsAssetHashes(it) }
            .flatten()

        return assetHashes
            .groupBy { it.campaignId }
            .mapValues {
                it.value
                    .filter { assetHashes -> assetHashes.bannerId != 0L }
                    .associateBy { assetHashes -> assetHashes.bannerId }
            }
    }

    private fun getChunkOfCampaignsAssetHashes(
        campaignIds: Collection<Long>,
    ): List<AssetHashes> {
        val ydbCampaignIds = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(campaignIds).keys

        val allAssetHashes = mutableListOf<AssetHashes>()
        var fromAdId = 0L
        var attemptsLimit = MAX_COUNT_OF_REQUESTS
        while (true) {
            val assetHashesFromYdb =
                uacYdbDirectAdRepository.getContentIdsByDirectCampaignIds(ydbCampaignIds, fromAdId, CHUNK_ADS_SIZE)
            allAssetHashes.addAll(assetHashesFromYdb)

            if (assetHashesFromYdb.size < CHUNK_ADS_SIZE) {
                break
            }
            if (attemptsLimit-- == 0) {
                logger.error("The limit of $MAX_COUNT_OF_REQUESTS attempts has been reached")
                break
            }
            fromAdId = assetHashesFromYdb.maxOf { it.bannerId }
        }

        val grutCampaignIds = campaignIds
            .filter { !ydbCampaignIds.contains(it) }
            .toSet()

        allAssetHashes.addAll(grutUacContentService.getAssetHashesByCampaignIds(grutCampaignIds))
        return allAssetHashes
    }
}
