package ru.yandex.direct.intapi.entity.bs.export.service;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.bs.export.model.TypeAndLogData;
import ru.yandex.direct.intapi.entity.bs.export.repository.BsExportLogsRepository;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.intapi.entity.bs.export.utils.FakeBsExportLogsUtils.BANNER;
import static ru.yandex.direct.intapi.entity.bs.export.utils.FakeBsExportLogsUtils.CONTEXT;
import static ru.yandex.direct.intapi.entity.bs.export.utils.FakeBsExportLogsUtils.ORDER;
import static ru.yandex.direct.intapi.entity.bs.export.utils.FakeBsExportLogsUtils.PRICE;
import static ru.yandex.direct.intapi.entity.bs.export.utils.FakeBsExportLogsUtils.fillResponse;
import static ru.yandex.direct.utils.HashingUtils.getSha256Hash;

/**
 * Сервис для получения логов отправки в БК (нужны в основном для тестов)
 * Логи берутся из дин таблицы yt, в которую пишет {@code BsExportLogsLbToYtJob}
 */
@Service
public class BsExportLogsService {

    private static final Logger logger = LoggerFactory.getLogger(BsExportLogsService.class);

    private static final String REQUEST = "request";
    private static final String RESPONSE = "response";

    private final BsExportLogsRepository bsExportLogsRepository;

    @Autowired
    public BsExportLogsService(BsExportLogsRepository bsExportLogsRepository) {
        this.bsExportLogsRepository = bsExportLogsRepository;
    }

    /**
     * Получение логов для {@param uuids}.
     *
     * @param omitMdsMeta - заменять ли MdsMeta на хэш
     */
    public Map<String, Map> getLogs(List<String> uuids, boolean omitMdsMeta) {
        Map<String, List<TypeAndLogData>> logs = bsExportLogsRepository.getLogs(uuids);
        Map<String, Map> result = new HashMap<>();

        for (String uuid : logs.keySet()) {
            ObjectNode resultNode = JsonUtils.getObjectMapper().createObjectNode();
            resultNode.putObject(REQUEST);
            resultNode.putObject(RESPONSE);

            for (TypeAndLogData typeAndLogData : logs.get(uuid)) {
                try {
                    JsonNode allData = JsonUtils.getObjectMapper().readTree(typeAndLogData.getLogData());
                    processRequestLogs((ObjectNode) resultNode.get(REQUEST), omitMdsMeta, typeAndLogData.getType(),
                            allData);
                } catch (IOException e) {
                    logger.error("Got {} while parsing {}", e, typeAndLogData.getLogData());
                }
            }

            fillResponse("root", resultNode.get(REQUEST), (ObjectNode) resultNode.get(RESPONSE));

            result.put(uuid, JsonUtils.getObjectMapper().convertValue(resultNode, Map.class));
        }
        return result;
    }

    private void processRequestLogs(ObjectNode requestNode, boolean omitMdsMeta, String level, JsonNode allData) {

        String cid = allData.get("cid").asText();
        String pid = allData.has("pid") ? allData.get("pid").asText() : null;
        JsonNode data = allData.get("data");

        if (level.equals(PRICE)) {
            if (!requestNode.has(cid)) {
                requestNode.putObject(cid);
            }
            ObjectNode cidNode = (ObjectNode) requestNode.get(cid);

            if (!cidNode.has(pid)) {
                cidNode.putArray(pid);
            }
            ArrayNode pidNode = (ArrayNode) cidNode.get(pid);
            pidNode.add(data);
            return;
        }

        if (!requestNode.has(ORDER)) {
            requestNode.putObject(ORDER);
        }
        ObjectNode orderNode = (ObjectNode) requestNode.get(ORDER);

        if (level.equals(ORDER)) {
            if (orderNode.has(cid)) {
                mergeNodes((ObjectNode) data, (ObjectNode) orderNode.get(cid));
            }
            orderNode.set(cid, data);
            ObjectNode campaignNode = (ObjectNode) orderNode.get(cid);
            if (!campaignNode.has(CONTEXT)) {
                campaignNode.putObject(CONTEXT);
            }
        }

        if (level.equals(CONTEXT)) {
            if (!orderNode.has(cid)) {
                orderNode.putObject(cid);
            }
            ObjectNode campaignNode = (ObjectNode) orderNode.get(cid);
            if (!campaignNode.has(CONTEXT)) {
                campaignNode.putObject(CONTEXT);
            }
            ObjectNode contextNode = (ObjectNode) campaignNode.get(CONTEXT);
            if (data.has(BANNER) && omitMdsMeta) {
                omitMdsMeta(data.get(BANNER));
            }
            if (data.has(BANNER)) {
                modifyBannerKeys(data.get(BANNER));
            }

            if (contextNode.has(pid)) {
                mergeNodes((ObjectNode) data, (ObjectNode) contextNode.get(pid));
            }
            contextNode.set(pid, data);
        }
    }

    private void mergeNodes(ObjectNode nodeTarget, ObjectNode nodeSource) {
        nodeSource.fieldNames().forEachRemaining(name -> {
            if (!nodeTarget.has(name)) {
                nodeTarget.set(name, nodeSource.get(name));
            }
        });
    }

    /**
     * Заменяем ключи у {@param banners} с "B_x_y_z" на "x"
     */
    private void modifyBannerKeys(JsonNode banners) {
        Map<String, JsonNode> newValues = new HashMap<>();

        Iterator<Map.Entry<String, JsonNode>> fields = banners.fields();

        while (fields.hasNext()) {
            Map.Entry<String, JsonNode> entry = fields.next();
            String key = entry.getKey();
            JsonNode node = entry.getValue();
            fields.remove();
            String cid = key.split("_")[2];
            newValues.put(cid, node);
        }

        ((ObjectNode) banners).setAll(newValues);
    }

    /**
     * Заменяем MdsMeta поля у баннеров в {@param banners} на хэш (чтобы занимало меньше места)
     */
    private void omitMdsMeta(JsonNode banners) {
        final String mdsMeta = "MdsMeta";

        for (JsonNode banner : banners) {

            for (String path : List.of("", "/Resources/ImagesInfo")) {

                JsonNode rootNode = banner.at(path);
                if (rootNode == null || !rootNode.isObject() || !rootNode.has(mdsMeta)) {
                    continue;
                }
                String mdsMetaString = rootNode.get(mdsMeta).asText();
                String hash = getSha256Hash(mdsMetaString);
                ((ObjectNode) rootNode).remove(mdsMeta);
                String newMdsMetaValue = String.format("MdsMeta omitted, hash of content is %s, see original data in " +
                        "YT logs by uuid", hash);
                ((ObjectNode) rootNode).put("MdsMetaStub", newMdsMetaValue);
            }
        }
    }
}
