package ru.yandex.direct.intapi.entity.campaigns.allowedpageids;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_PROD;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_TEST;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_PROD;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_TEST;

@RestController
@Api(value = "Контроллер для управления площадками, на которых допустимо показывать материалы кампании")
@RequestMapping(value = "/campaigns/page_ids")
@AllowServices(production = {GEO_PRODUCT_PROD, SURVEYS_PROD},
        testing = {GEO_PRODUCT_TEST, DIRECT_DEVELOPER, SURVEYS_TEST})
public class CampaignAllowedPageIdsController {
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final CampaignAllowedPageIdsValidation validationService;

    @Autowired
    public CampaignAllowedPageIdsController(
            CampaignRepository campaignRepository,
            ShardHelper shardHelper,
            CampaignAllowedPageIdsValidation validationService
    ) {
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.validationService = validationService;
    }

    @ApiOperation(
            httpMethod = "GET",
            value = "Получить список площадок на которых разрешены показы материалов компании",
            notes = "Возвращает словарь { cid -> [allowed_page_ids, ..], .. }, если кампания не найдена, "
                    + "она не будет присутствовать в ответе, если найдена, "
                    + "но page_ids для нее не заданы, для этой кампании в ответе вернется пустой список"
    )
    @RequestMapping(path = "",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public Map<Long, List<Long>> getCampaignsAllowedPageIds(
            @ApiParam(value = "Список Id кампаний",
                    required = true)
            @RequestParam @Nonnull List<String> campaignIds
    ) {
        List<Long> validIds = validationService.checkAndParseCampaignIds(campaignIds);

        ShardedData<Long> cidsByShard = shardHelper.groupByShard(validIds, ShardKey.CID);

        Map<Long, List<Long>> campaignsAllowedPageIds = new HashMap<>();

        cidsByShard.forEach((shard, ids) -> {
            campaignsAllowedPageIds.putAll(campaignRepository.getCampaignsAllowedPageIds(shard, ids));
            ids.forEach(cid -> campaignsAllowedPageIds.putIfAbsent(cid, Collections.emptyList()));
        });

        return campaignsAllowedPageIds;
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Задать на кампанию список площадок на которых разрешено показывать материалы этой кампании",
            notes = "Если список площадок не передан, то он сбрасывается,"
                    + " т.е. материалы кампании можно будет показывать везде"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 404, message = "Not found", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "OK")
            }
    )
    @RequestMapping(
            path = "",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    public HttpStatus setCampaignAllowedPageIds(
            @ApiParam(value = "Id кампании", required = true)
            @RequestParam @Nonnull String campaignId,
            @ApiParam(value = "Список Id площадок")
            @RequestBody(required = false) List<String> pageIdsRequest
    ) {
        Long validCampaignId = validationService.checkAndParseCampaignId(campaignId);

        int shard = shardHelper.getShardByCampaignId(validCampaignId);
        if (shard == NO_SHARD) {
            throw new IntApiException(
                    HttpStatus.NOT_FOUND,
                    new ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "Not found"));
        }

        List<Long> pageIds = validationService.checkAndParsePageIds(pageIdsRequest);

        campaignRepository.updateAllowedPageIdsAndFlushStatusBsSynced(shard, validCampaignId, pageIds);

        return HttpStatus.OK;
    }

}

