package ru.yandex.direct.intapi.entity.campaigns.allowedpageids;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.IntApiDefects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.isLong;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.listSize;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyString;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNegativeWholeNumber;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.ValidationUtils.getErrorText;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
class CampaignAllowedPageIdsValidation {
    private static final int MAX_CAMPAIGN_IDS = 10_000;
    private static final int MAX_PAGE_IDS = 1_000;

    Long checkAndParseCampaignId(String campaignIdRequest) {
        checkCampaignId(campaignIdRequest);
        return Long.parseLong(campaignIdRequest);
    }

    List<Long> checkAndParseCampaignIds(List<String> campaignIdsRequest) {
        checkIds("campaign_ids", 1, MAX_CAMPAIGN_IDS, campaignIdsRequest);
        return campaignIdsRequest.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    List<Long> checkAndParsePageIds(List<String> pageIdsRequest) {
        if (pageIdsRequest == null) {
            return Collections.emptyList();
        }

        checkIds("page_ids", 0, MAX_PAGE_IDS, pageIdsRequest);
        return pageIdsRequest.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void checkCampaignId(String id) {
        checkIds("campaign_id", 1, 1, Collections.singletonList(id));
    }

    private void checkIds(String paramName, int minSize, int maxSize, List<String> ids) {
        ValidationResult<List<String>, IntApiDefect> validateStringIds = validateRequestIds(minSize, maxSize, ids);
        if (validateStringIds.hasAnyErrors()) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    new ErrorResponse(
                            ErrorResponse.ErrorCode.BAD_PARAM,
                            getErrorText(validateStringIds, path(PathHelper.field(paramName)))));
        }
    }

    private ValidationResult<List<String>, IntApiDefect> validateRequestIds(
            int minSize, int maxSize,
            List<String> ids
    ) {
        return ListValidationBuilder.of(ids, IntApiDefect.class)
                .check(notNull())
                .check(listSize(minSize, maxSize))
                .checkEach(notNull())
                .checkEach(notEmptyString(), When.isValid())
                .checkEach(notNegativeWholeNumber(), When.isValid())
                .checkEach(isLong(), When.isValid())
                .checkEach(Constraint.fromPredicate(s -> Long.parseLong(s) > 0, IntApiDefects.notNullString()),
                        When.isValid())
                .getResult();
    }

}
