package ru.yandex.direct.intapi.entity.campaigns.controller;

import java.util.Arrays;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.campaigns.model.CampaignsAddZenRequest;
import ru.yandex.direct.intapi.entity.campaigns.model.CampaignsAddZenResponse;
import ru.yandex.direct.intapi.entity.campaigns.model.CampaignsAddZenValidationError;
import ru.yandex.direct.intapi.entity.campaigns.service.CampaignsAddService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.ZEN_PROD;
import static ru.yandex.direct.tvm.TvmService.ZEN_TEST;

@RestController
@Api(value = "Контроллер для создания кампаний")
@RequestMapping(value = "/campaigns/add")
public class CampaignsAddController {
    private static final Logger LOGGER = LoggerFactory.getLogger(CampaignsAddController.class);

    private final ClientService clientService;
    private final CampaignsAddService campaignsAddService;

    @Autowired
    public CampaignsAddController(ClientService clientService,
            CampaignsAddService campaignsAddService) {
        this.clientService = clientService;
        this.campaignsAddService = campaignsAddService;
    }

    /**
     * Создаём кампанию для Дзена под конкретного клиента с кошельком. Кампания без групп,
     * её параметры в основном не имеют значения, т. к. крутиться будет самим Дзеном, и нужна только для указания трат.
     * Кампания должна быть годна для отправки в БК с особым source, contentType (вычисляется транспортом) и в Баланс.
     * @param request — JSON-словарь с ClientID
     * @return HTTP-ответ с номером кампании или ошибкой с описанием
     */
    @AllowServices(
            production = {ZEN_PROD},
            testing = {ZEN_TEST, DIRECT_DEVELOPER}
    )
    @ApiOperation(value = "Создать минимальную кампанию для Дзена, вернуть её номер")
    @PostMapping(value = "zen")
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad request", response = CampaignsAddZenResponse.class),
                    @ApiResponse(code = 500, message = "Internal Server Error",
                            response = CampaignsAddZenResponse.class),
                    @ApiResponse(code = 200, message = "OK", response = CampaignsAddZenResponse.class),
            }
    )
    public ResponseEntity<CampaignsAddZenResponse> addZenCampaign(@RequestBody CampaignsAddZenRequest request) {
        var clientId = ClientId.fromLong(request.getClientId());
        var client = clientService.getClient(clientId);
        if (client == null) {
            return handleAddZenCampaignValidationError(CampaignsAddZenValidationError.NO_CLIENT);
        }
        if (!client.getRole().equals(RbacRole.CLIENT)) {
            return handleAddZenCampaignValidationError(CampaignsAddZenValidationError.NOT_A_CLIENT);
        }
        var uid = client.getChiefUid();
        // не все операцию могут быть готовы к фиктивному оператору (uid=0), используем подходящего настоящего оператора
        var operatorUid = client.getAgencyUserId() == null ? uid : client.getAgencyUserId();
        if (!campaignsAddService.doesClientHaveWallet(client, operatorUid)) {
            return handleAddZenCampaignValidationError(CampaignsAddZenValidationError.NO_WALLET);
        }

        var lastCampaignId = campaignsAddService.getLastZenCampaignId(client);
        if (lastCampaignId.isPresent()) {
            return hasZenCampaignResponse(lastCampaignId.get());
        }

        var massResult = campaignsAddService.createZenTextCampaign(operatorUid, client);
        return handleAddZenCampaignResult(massResult);
    }

    private ResponseEntity<CampaignsAddZenResponse> handleAddZenCampaignResult(MassResult<Long> massResult) {
        var response = new CampaignsAddZenResponse();
        if (!massResult.getResult().isEmpty() && massResult.get(0).isSuccessful()) {
            response.setCampaignId(massResult.get(0).getResult());
            campaignsAddService.moveToQueue(massResult.get(0).getResult());
            return new ResponseEntity<>(response, HttpStatus.OK);
        }

        response.setError("OPERATION_ERROR");
        if (massResult.getResult().isEmpty()) {
            LOGGER.error("Неизвестная причина падения");
            response.setErrorDetails("Нет информации о падении");
        } else {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("Не удалось создать кампанию: {}",
                        Arrays.toString(massResult.get(0).getErrors().toArray()));
            }
            response.setErrorDetails(massResult.get(0).getErrors().get(0).toString());
        }
        return new ResponseEntity<>(response, HttpStatus.INTERNAL_SERVER_ERROR);
    }

    private ResponseEntity<CampaignsAddZenResponse> handleAddZenCampaignValidationError(
            CampaignsAddZenValidationError error) {
        var response = new CampaignsAddZenResponse()
                .withError(error.name())
                .withErrorDetails(error.getDescription());
        return new ResponseEntity<>(response, HttpStatus.BAD_REQUEST);
    }

    // закидываем замечание в error, чтобы не сильно менять модель ответа
    private ResponseEntity<CampaignsAddZenResponse> hasZenCampaignResponse(
            Long campaignId) {
        var error = CampaignsAddZenValidationError.HAS_ZEN_CAMPAIGN;
        var response = new CampaignsAddZenResponse()
                .withCampaignId(campaignId)
                .withError(error.name())
                .withErrorDetails(error.getDescription());
        return new ResponseEntity<>(response, HttpStatus.OK);
    }
}
