package ru.yandex.direct.intapi.entity.campaigns.impressionrate;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithImpressionRate;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.campaigns.impressionrate.model.CampaignImpressionRateGetResponse;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIdsFilter;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_PROD;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_TEST;

@RestController
@Api(value = "Контроллер для управления ограничением частоты показов")
@RequestMapping(value = "/campaigns/impression_rate")
@AllowServices(production = {SURVEYS_PROD}, testing = {DIRECT_DEVELOPER, SURVEYS_TEST})
public class CampaignImpressionRateController {
    private final ShardHelper shardHelper;
    private final ValidationResultConversionService validationResultConversionService;
    private final CampaignOperationService campaignOperationService;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public CampaignImpressionRateController(
            ShardHelper shardHelper,
            ValidationResultConversionService validationResultConversionService,
            CampaignOperationService campaignOperationService,
            CampaignTypedRepository campaignTypedRepository
    ) {
        this.shardHelper = shardHelper;
        this.validationResultConversionService = validationResultConversionService;
        this.campaignOperationService = campaignOperationService;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    @ApiOperation(
            httpMethod = "GET",
            value = "Получить ограничение частоты показов на кампанию"
    )
    @RequestMapping(
            path = "",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON)
    @ResponseBody
    public CampaignImpressionRateGetResponse getCampaignImpressionRate(
            @ApiParam(value = "Id клиента", required = true)
            @RequestParam(value = "client_id") @Nonnull Long rawClientId,
            @ApiParam(value = "Id кампании", required = true)
            @RequestParam(value = "campaign_id") @Nonnull Long campaignId
    ) {
        BaseCampaign campaign = getCampaign(campaignId);
        if (checkIfCampaignNotFound(campaign, rawClientId)) {
            return new CampaignImpressionRateGetResponse(
                    validationResultConversionService.buildValidationResponse(
                            ValidationResult.failed(campaignId, campaignNotFound())),
                    null, null);
        }

        CampaignWithImpressionRate campaignWithRate = (CampaignWithImpressionRate) campaign;
        return new CampaignImpressionRateGetResponse(
                validationResultConversionService.buildValidationResponse(ValidationResult.success(campaignId)),
                campaignWithRate.getImpressionRateCount(), campaignWithRate.getImpressionRateIntervalDays());
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Задать на кампанию ограничение частоты показов",
            notes = "За определенный период времени баннер будет показан одному и тому же пользователю не более N раз"
    )
    @RequestMapping(
            path = "",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON)
    @ResponseBody
    public IntapiValidationResponse setCampaignImpressionRate(
            @ApiParam(value = "Id клиента", required = true)
            @RequestParam(value = "client_id") @Nonnull Long rawClientId,
            @ApiParam(value = "Id кампании", required = true)
            @RequestParam(value = "campaign_id") @Nonnull Long campaignId,
            @ApiParam(value = "Максимальное количество показов за период")
            @RequestParam(value = "rate_count", required = false) Integer rateCount,
            @ApiParam(value = "Продолжительность периода в днях")
            @RequestParam(value = "rate_interval_days", required = false) Integer rateIntervalDays
    ) {
        BaseCampaign campaign = getCampaign(campaignId);
        if (checkIfCampaignNotFound(campaign, rawClientId)) {
            return validationResultConversionService.buildValidationResponse(
                    ValidationResult.failed(campaignId, campaignNotFound()));
        }
        ClientId clientId = ClientId.fromLong(rawClientId);
        Long operatorUid = ((CommonCampaign) campaign).getUid();

        ModelChanges<CampaignWithImpressionRate> modelChanges =
                new ModelChanges<>(campaignId, CampaignWithImpressionRate.class)
                        .process(rateCount, CampaignWithImpressionRate.IMPRESSION_RATE_COUNT)
                        .process(rateIntervalDays, CampaignWithImpressionRate.IMPRESSION_RATE_INTERVAL_DAYS);

        var options = new CampaignOptions();
        var updateCampaignOperation = campaignOperationService
                .createRestrictedCampaignUpdateOperation(
                        Collections.singletonList(modelChanges), operatorUid, UidAndClientId.of(operatorUid, clientId),
                        options);

        MassResult<Long> operationResult = updateCampaignOperation.apply();

        return validationResultConversionService.buildValidationResponse(operationResult.getValidationResult());
    }

    @Nullable
    private BaseCampaign getCampaign(Long campaignId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        List<? extends BaseCampaign> campaigns = shard != NO_SHARD ?
                campaignTypedRepository.getSafely(
                        shard, campaignIdsFilter(Collections.singletonList(campaignId)),
                        List.of(CampaignWithImpressionRate.class, CommonCampaign.class))
                : Collections.emptyList();

        return campaigns.isEmpty() ? null : campaigns.get(0);
    }

    private static boolean checkIfCampaignNotFound(@Nullable BaseCampaign campaign, Long clientId) {
        return !(campaign instanceof CampaignWithImpressionRate) || !(campaign instanceof CommonCampaign) ||
                !((CommonCampaign) campaign).getClientId().equals(clientId);
    }
}
