package ru.yandex.direct.intapi.entity.campaigns.language;

import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithContentLanguage;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithContentLanguageValidator;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_MARKETING_PROD;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_MARKETING_TEST;

@RestController
@Api(value = "Контроллер для управления языком кампании")
@RequestMapping(value = "/campaigns/language")
@AllowServices(production = {EDA_RESTAPP_MARKETING_PROD},
        testing = {EDA_RESTAPP_MARKETING_TEST, DIRECT_DEVELOPER})
public class CampaignLanguageController {

    private final CampaignRepository campaignRepository;
    private final GeoTreeFactory geoTreeFactory;
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public CampaignLanguageController(
            CampaignRepository campaignRepository,
            GeoTreeFactory geoTreeFactory,
            ShardHelper shardHelper,
            AdGroupRepository adGroupRepository,
            ValidationResultConversionService validationResultConversionService
    ) {
        this.campaignRepository = campaignRepository;
        this.geoTreeFactory = geoTreeFactory;
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.validationResultConversionService = validationResultConversionService;
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Установить язык кампании"
    )
    @RequestMapping(
            path = "",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    public IntapiValidationResponse setCampaignContentLanguage(
            @ApiParam(value = "Id кампании", required = true)
            @RequestParam @Nonnull Long campaignId,
            @ApiParam(value = "Язык кампании")
            @RequestParam @Nullable ContentLanguage language
    ) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        if (shard == NO_SHARD) {
            IntapiValidationResult intapiValidationResult = validationResultConversionService
                    .buildIntapiValidationResult(ValidationResult.failed(campaignId, campaignNotFound()));
            return new IntapiValidationResponse(intapiValidationResult);
        }

        ModelChanges<CampaignWithContentLanguage> modelChanges =
                ModelChanges.build(campaignId, CampaignWithContentLanguage.class,
                        CampaignWithContentLanguage.CONTENT_LANGUAGE, language);

        var vr = validateContentLanguage(shard, modelChanges);

        if (!vr.hasAnyErrors()) {
            campaignRepository.updateCampaignLang(shard, List.of(campaignId), language);
        }

        return new IntapiValidationResponse(validationResultConversionService.buildIntapiValidationResult(vr));
    }

    private ValidationResult<ModelChanges<CampaignWithContentLanguage>, Defect> validateContentLanguage(
            int shard,
            ModelChanges<CampaignWithContentLanguage> modelChanges
    ) {
        Map<Long, List<AdGroupSimple>> adGroups =
                adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, List.of(modelChanges.getId()));
        var validator = new CampaignWithContentLanguageValidator(getRussianGeoTree(), adGroups, true);
        return validator.apply(modelChanges);
    }

    private GeoTree getRussianGeoTree() {
        // Для валидации соответствия регионов с языком кампании и баннеров используется российское геодерево
        return geoTreeFactory.getRussianGeoTree();
    }
}
