package ru.yandex.direct.intapi.entity.campaigns.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.bs.export.queue.service.BsExportQueueService;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.timetarget.TimeTargetUtils;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.model.StrategyName.DEFAULT_;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.defaultTimeTarget;

@Service
@ParametersAreNonnullByDefault
public class CampaignsAddService {
    private static final String ZEN_CAMPAIGN_NAME = "Дзен";
    private static final Duration PROPERTY_CACHING_DURATION = Duration.ofMinutes(1L);

    private final CampaignOperationService campaignOperationService;
    private final WalletService walletService;
    private final UserService userService;
    private final ShardHelper shardHelper;
    private final BsExportQueueService bsExportQueueService; // на время тестирования интеграции
    private final CampaignService campaignService;

    private final PpcProperty<String> bsQueueProperty; // на время тестирования интеграции

    @Autowired
    public CampaignsAddService(CampaignOperationService campaignOperationService,
            WalletService walletService,
            UserService userService,
            ShardHelper shardHelper,
            PpcPropertiesSupport ppcPropertiesSupport,
            BsExportQueueService bsExportQueueService,
            CampaignService campaignService) {
        this.campaignOperationService = campaignOperationService;
        this.walletService = walletService;
        this.userService = userService;
        this.shardHelper = shardHelper;
        bsQueueProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.ZEN_BS_TRANSPORT_QUEUE_ON_CREATION, PROPERTY_CACHING_DURATION
        );
        this.bsExportQueueService = bsExportQueueService;
        this.campaignService = campaignService;
    }

    public MassResult<Long> createZenTextCampaign(Long operatorUid, Client client) {
        var options = new CampaignOptions.Builder()
                .withKeepModerationStatuses(true)
                .build();
        var uidAndClientId = UidAndClientId.of(client.getChiefUid(), ClientId.fromLong(client.getClientId()));
        var user = userService.getUser(operatorUid);
        if (user == null) { // у нас в базе что-то не так, не должно случаться
            return MassResult.brokenMassAction(null,
                    ValidationResult.failed(null, new Defect<>(DefectIds.INCONSISTENT_STATE)));
        }
        var email = user.getRecommendationsEmail() != null ? user.getRecommendationsEmail() : user.getEmail();
        var campaign = getZenTextCampaign(email);

        var addCampaignOperation = campaignOperationService.createRestrictedCampaignAddOperation(
                List.of(campaign), operatorUid, uidAndClientId, options
        );
        return addCampaignOperation.prepareAndApply();
    }

    /**
     * Проверяем, что у клиента есть кошелёк
     * @param client — клиент, у которого проверяем
     * @param operatorUid — uid оператора, которым планируем добавлять кампанию
     * @return есть кошелёк или нет кошелька
     */
    public boolean doesClientHaveWallet(Client client, Long operatorUid) {
        var clientId = ClientId.fromLong(client.getClientId());
        var uid = client.getChiefUid();
        var agencyUidAndClientId = userService.getAgencyUidAndClientId(
                operatorUid, client.getAgencyUserId(), client.getAgencyClientId()
        );
        var addCampaignParametersContainer = RestrictedCampaignsAddOperationContainer.create(
                shardHelper.getShardByClientIdStrictly(clientId), operatorUid, clientId, uid, uid
        );

        var wallet = walletService.getWalletForNewCampaigns(addCampaignParametersContainer, agencyUidAndClientId);
        return wallet != null;
    }

    // вернуть id последней созданной кампании Дзена у клиента
    public Optional<Long> getLastZenCampaignId(Client client) {
        var clientId = ClientId.fromLong(client.getClientId());
        ClientId agencyClientId = client.getAgencyClientId() == null ?
                null : ClientId.fromLong(client.getAgencyClientId());
        return campaignService.getLastNotArchivedCampaignOfSource(clientId, agencyClientId, CampaignSource.ZEN);
    }

    private static TextCampaign getZenTextCampaign(String email) {
        // большинство параметров задаётся просто, чтобы были. На показ в БК влиять не должны.
        var dbStrategy = (DbStrategy) new DbStrategy()
                .withStrategy(CampOptionsStrategy.DIFFERENT_PLACES)
                .withPlatform(CampaignsPlatform.BOTH)
                .withStrategyName(DEFAULT_)
                .withPlatform(CampaignsPlatform.CONTEXT)
                .withAutobudget(CampaignsAutobudget.NO)
                .withStrategyData(new StrategyData().withVersion(1L).withName("default"));
        return new TextCampaign()
                .withSource(CampaignSource.ZEN)
                .withStatusModerate(CampaignStatusModerate.YES)
                .withStatusPostModerate(CampaignStatusPostmoderate.ACCEPTED)
                .withName(ZEN_CAMPAIGN_NAME)
                .withType(CampaignType.TEXT)
                .withEmail(email)
                .withStartDate(LocalDateTime.now().toLocalDate())
                .withStrategy(dbStrategy)
                .withTimeTarget(defaultTimeTarget())
                .withHasExtendedGeoTargeting(true)
                .withGeo(Set.of(0))
                .withSmsTime(new TimeInterval().withStartHour(0).withStartMinute(0).withEndHour(0).withEndMinute(0))
                // это бы не задавать, DIRECT-141085
                .withBroadMatch(new BroadMatch().withBroadMatchFlag(false).withBroadMatchLimit(0))
                .withDayBudget(BigDecimal.ZERO)
                .withDayBudgetShowMode(DayBudgetShowMode.DEFAULT_)
                .withEnablePausedByDayBudgetEvent(true)
                .withEnableSendAccountNews(false)
                .withTimeZoneId(TimeTargetUtils.DEFAULT_TIMEZONE)
                .withSmsFlags(EnumSet.noneOf(SmsFlag.class))
                .withDisabledSsp(List.of())
                .withAttributionModel(CampaignAttributionModel.LAST_YANDEX_DIRECT_CLICK)
                .withEnableCheckPositionEvent(false)
                .withHasSiteMonitoring(false)
                .withEnableCpcHold(false)
                .withHasAddMetrikaTagToUrl(true)
                .withHasAddOpenstatTagToUrl(false)
                .withHasTitleSubstitution(true)
                .withMetrikaCounters(List.of())
                .withExcludePausedCompetingAds(false)
                .withEnableCompanyInfo(false)
                .withPlacementTypes(Set.of())
                .withIsRecommendationsManagementEnabled(false)
                .withIsPriceRecommendationsManagementEnabled(false)
                .withIsTouch(false);
    }

    /**
     * Подвинуть кампанию в разработческую очередь, указанную в проперти ZEN_BS_TRANSPORT_QUEUE_ON_CREATION
     * @param campaignId — номер кампании
     */
    public void moveToQueue(Long campaignId) {
        var queue = bsQueueProperty.get();
        if (queue == null) {
            return;
        }

        QueueType queueType;
        switch (queue) {
            case "DEV1":
                queueType = QueueType.DEV1;
                break;
            case "DEV2":
                queueType = QueueType.DEV2;
                break;
            default:
                queueType = QueueType.NOSEND; // положим сюда, потом разберёмся, почему не сматчилось.
        }
        bsExportQueueService.setQueueType(queueType, List.of(campaignId));
    }
}
