package ru.yandex.direct.intapi.entity.clientofflinereport

import com.fasterxml.jackson.annotation.JsonProperty
import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReport
import ru.yandex.direct.core.entity.postviewofflinereport.service.PostviewOfflineReportService
import ru.yandex.direct.core.entity.postviewofflinereport.validation.PostviewOfflineReportConstraints.Companion.MAX_PROCESSING_REPORTS_COUNT
import ru.yandex.direct.core.entity.postviewofflinereport.validation.PostviewOfflineReportConstraints.Companion.isValidDate
import ru.yandex.direct.dbqueue.model.DbQueueJobStatus
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.intapi.ErrorResponse
import ru.yandex.direct.intapi.IntApiException
import ru.yandex.direct.intapi.logging.ClientIdParam
import ru.yandex.direct.tvm.AllowServices
import ru.yandex.direct.tvm.TvmService
import ru.yandex.direct.web.core.model.WebResponse
import ru.yandex.direct.web.core.model.WebSuccessResponse
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import javax.ws.rs.GET
import javax.ws.rs.POST

@RestController
@Api(value = "API для работы с клиентскими оффлайн-отчетами")
@RequestMapping("/client_offline_report/postview", produces = [MediaType.APPLICATION_JSON_VALUE])
@AllowServices(
    production = [TvmService.DIRECT_WEB_PROD, TvmService.DIRECT_INTAPI_PROD],
    testing = [TvmService.DIRECT_WEB_TEST, TvmService.DIRECT_INTAPI_TEST, TvmService.DIRECT_DEVELOPER],
    sandboxTesting = [TvmService.DIRECT_DEVELOPER]
)
class PostviewOfflineReportController(
    private val postviewOfflineReportService: PostviewOfflineReportService,
    private val shardHelper: ShardHelper,
) {
    companion object {
        private val logger: Logger = LoggerFactory.getLogger(PostviewOfflineReportController::class.java)

        data class CreatePostviewOfflineReportRequest(
            @JsonProperty("operator_uid") val operatorUid: Long,
            @JsonProperty("cids") val campaignIds: Set<Long>,
            @JsonProperty("date_from") val dateFrom: LocalDate,
            @JsonProperty("date_to") val dateTo: LocalDate,
        )

        data class DeletePostviewOfflineReportRequest(
            @JsonProperty("client_id") val clientId: ClientId, @JsonProperty("report_id") val reportId: Long
        )

        data class CreatePostviewOfflineReportResponse(
            val reportId: Long? = null,
            val message: String? = null,
        ) : WebResponse {
            override fun isSuccessful(): Boolean {
                return reportId != null
            }
        }

        data class GetPostviewOfflineReportsResponse(
            val reports: Set<PostviewOfflineReportData> = setOf(),
        )

        data class PostviewOfflineReportData(
            @JsonProperty("report_id") val reportId: Long,
            @JsonProperty("cids") val cids: Set<Long>,
            @JsonProperty("date_from") val dateFrom: String,
            @JsonProperty("date_to") val dateTo: String,
            @JsonProperty("status") val status: DbQueueJobStatus,
            @JsonProperty("create_time") val createTime: String,
            @JsonProperty("finish_time") val finishTime: String?,
            @JsonProperty("error") val error: String?,
            @JsonProperty("report_url") val reportUrl: String?
        )

        private val formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")

        @JvmStatic
        fun convertReportToReportData(report: PostviewOfflineReport): PostviewOfflineReportData {
            return PostviewOfflineReportData(
                reportId = report.id,
                cids = report.campaignIds,
                dateFrom = report.dateFrom.toString(),
                dateTo = report.dateTo.toString(),
                createTime = report.createTime.format(formatter),
                finishTime = report.finishTime?.format(formatter),
                status = report.status,
                error = report.error,
                reportUrl = report.url
            )
        }
    }

    @POST
    @ApiOperation(
        value = "валидация параметров postview-отчета",
        httpMethod = "POST",
        nickname = "validatePostviewReport",
    )
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse::class),
    )
    @PostMapping(
        value = ["validate"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @ResponseBody
    fun validatePostviewOfflineReport(
        @RequestBody createReportRequest: CreatePostviewOfflineReportRequest
    ): WebResponse {
        val clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(createReportRequest.campaignIds.first()))

        val reports = postviewOfflineReportService.getReportList(clientId)
        val processingReportsCount =
            reports.filter { it.status in (setOf(DbQueueJobStatus.NEW, DbQueueJobStatus.GRABBED)) }.size
        if (processingReportsCount >= MAX_PROCESSING_REPORTS_COUNT) {
            throw badParamsException("processing_reports_limit_reached")
        }

        if (!isValidDate(createReportRequest.dateFrom)) {
            throw badParamsException("invalid_date_from")
        }

        if (!isValidDate(createReportRequest.dateTo)) {
            throw badParamsException("invalid_date_to")
        }

        if (createReportRequest.dateTo.isBefore(createReportRequest.dateFrom)) {
            throw badParamsException("date_to_before_date_from")
        }

        if (isReportExists(reports, createReportRequest)) {
            throw badParamsException("report_exists")
        }

        return WebSuccessResponse()
    }

    private fun isReportExists(
        reports: Set<PostviewOfflineReport>, request: CreatePostviewOfflineReportRequest
    ): Boolean {
        return reports.any {
            it.campaignIds == request.campaignIds && it.dateFrom == request.dateFrom && it.dateTo == request.dateTo
        }
    }

    private fun badParamsException(message: String): IntApiException {
        return IntApiException(
            HttpStatus.BAD_REQUEST, ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, message)
        )
    }

    @POST
    @ApiOperation(
        value = "создать задание на формирование postview-отчета",
        httpMethod = "POST",
        nickname = "createPostviewReport",
    )
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = CreatePostviewOfflineReportResponse::class),
    )
    @PostMapping(
        value = ["create"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @ResponseBody
    fun createPostviewOfflineReport(
        @RequestBody createReportRequest: CreatePostviewOfflineReportRequest
    ): CreatePostviewOfflineReportResponse {
        val clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(createReportRequest.campaignIds.first()))
        val reports = postviewOfflineReportService.getReportList(clientId)
        if (isReportExists(reports, createReportRequest)) {
            throw badParamsException("report_exists")
        }
        val report = postviewOfflineReportService.createReport(
            createReportRequest.operatorUid,
            clientId,
            createReportRequest.campaignIds,
            createReportRequest.dateFrom,
            createReportRequest.dateTo
        )

        return CreatePostviewOfflineReportResponse(report.id)
    }

    @POST
    @ApiOperation(
        value = "удалить задание на формирование postview-отчета",
        httpMethod = "POST",
        nickname = "deletePostviewReport",
    )
    @ApiResponses(
        ApiResponse(code = 500, message = "Error", response = ErrorResponse::class),
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse::class),
    )
    @PostMapping(
        value = ["delete"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @ResponseBody
    fun deletePostviewOfflineReport(
        @RequestBody deleteReportRequest: DeletePostviewOfflineReportRequest
    ): WebResponse {
        val report = postviewOfflineReportService.getReport(deleteReportRequest.clientId, deleteReportRequest.reportId)
            ?: throw IntApiException(
                HttpStatus.NOT_FOUND, ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "job not found")
            )

        if (!report.isDeletable()) {
            throw IntApiException(
                HttpStatus.INTERNAL_SERVER_ERROR,
                ErrorResponse(
                    ErrorResponse.ErrorCode.INTERNAL_ERROR,
                    "invalid report (${report.id}) status (${report.status}) for delete"
                )
            )
        }

        val isDeleted =
            postviewOfflineReportService.deleteReport(deleteReportRequest.clientId, deleteReportRequest.reportId)
        if (!isDeleted) {
            throw IntApiException(
                HttpStatus.INTERNAL_SERVER_ERROR,
                ErrorResponse(ErrorResponse.ErrorCode.INTERNAL_ERROR, "can't delete report (${report.id})")
            )
        }

        return WebSuccessResponse()
    }

    @GET
    @ApiOperation(
        value = "получить список заданий на формирование postview-отчетов",
        httpMethod = "GET",
        nickname = "getPostviewReportList",
    )
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = GetPostviewOfflineReportsResponse::class),
    )
    @GetMapping(
        value = ["get_list"],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @ResponseBody
    fun getPostviewOfflineReportList(
        @RequestParam("client_id") @ClientIdParam clientId: Long
    ): GetPostviewOfflineReportsResponse {
        val reports = postviewOfflineReportService.getReportList(ClientId.fromLong(clientId))

        return GetPostviewOfflineReportsResponse(
            reports.map {
                convertReportToReportData(it)
            }.toSet()
        )
    }

    @GET
    @ApiOperation(
        value = "получить указанное задание на формирование postview-отчетов",
        httpMethod = "GET",
        nickname = "getPostviewReport",
    )
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = ErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = PostviewOfflineReportData::class),
    )
    @GetMapping(
        value = ["get"],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    @ResponseBody
    fun getPostviewOfflineReport(
        @RequestParam("client_id") @ClientIdParam clientId: Long, @RequestParam("report_id") reportId: Long
    ): PostviewOfflineReportData {
        val report =
            postviewOfflineReportService.getReport(ClientId.fromLong(clientId), reportId) ?: throw IntApiException(
                HttpStatus.NOT_FOUND, ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "job not found")
            )
        return convertReportToReportData(report)
    }
}
